<?php

namespace App\Notifications;

use App\Models\InvoiceInstallment;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class InstallmentDueNotification extends Notification implements ShouldQueue
{
    use Queueable;

    protected $installment;
    protected $daysRemaining;

    /**
     * Create a new notification instance.
     *
     * @param InvoiceInstallment $installment
     * @param int $daysRemaining
     * @return void
     */
    public function __construct(InvoiceInstallment $installment, int $daysRemaining)
    {
        $this->installment = $installment;
        $this->daysRemaining = $daysRemaining;
    }

    /**
     * Get the notification's delivery channels.
     *
     * @param  mixed  $notifiable
     * @return array
     */
    public function via($notifiable)
    {
        return ['mail', 'database'];
    }

    /**
     * Get the mail representation of the notification.
     *
     * @param  mixed  $notifiable
     * @return \Illuminate\Notifications\Messages\MailMessage
     */
    public function toMail($notifiable)
    {
        $invoice = $this->installment->invoice;
        $customer = $invoice->customer;
        $subject = $this->daysRemaining > 0
            ? "تذكير: قسط مستحق خلال {$this->daysRemaining} يوم"
            : "تنبيه: قسط متأخر عن موعد استحقاقه";

        return (new MailMessage)
            ->subject($subject)
            ->greeting("مرحباً {$customer->name}")
            ->line($this->daysRemaining > 0
                ? "نود تذكيركم بأن لديكم قسط مستحق خلال {$this->daysRemaining} يوم."
                : "نود إعلامكم بأن لديكم قسط متأخر عن موعد استحقاقه.")
            ->line("رقم الفاتورة: {$invoice->invoice_number}")
            ->line("تاريخ الاستحقاق: {$this->installment->due_date->format('Y-m-d')}")
            ->line("المبلغ المستحق: " . number_format($this->installment->amount - $this->installment->paid_amount, 2) . " ريال")
            ->action('عرض تفاصيل الفاتورة', url("/invoices/{$invoice->id}"))
            ->line('شكراً لتعاونكم معنا.');
    }

    /**
     * Get the array representation of the notification.
     *
     * @param  mixed  $notifiable
     * @return array
     */
    public function toArray($notifiable)
    {
        $invoice = $this->installment->invoice;
        $customer = $invoice->customer;

        return [
            'title' => $this->daysRemaining > 0
                ? "تذكير: قسط مستحق خلال {$this->daysRemaining} يوم"
                : "تنبيه: قسط متأخر عن موعد استحقاقه",
            'message' => $this->daysRemaining > 0
                ? "لديكم قسط مستحق للفاتورة رقم {$invoice->invoice_number} خلال {$this->daysRemaining} يوم."
                : "لديكم قسط متأخر للفاتورة رقم {$invoice->invoice_number} عن موعد استحقاقه.",
            'invoice_id' => $invoice->id,
            'installment_id' => $this->installment->id,
            'customer_id' => $customer->id,
            'amount' => $this->installment->amount - $this->installment->paid_amount,
            'due_date' => $this->installment->due_date->format('Y-m-d'),
            'days_remaining' => $this->daysRemaining,
            'type' => 'installment_due',
            'icon' => $this->daysRemaining > 0 ? 'fas fa-clock' : 'fas fa-exclamation-circle',
            'color' => $this->daysRemaining > 0 ? 'warning' : 'danger',
        ];
    }
}
