<?php

namespace App\Notifications;

use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;
use Illuminate\Support\Collection;
use App\Models\InventoryAlert;

class ExpiryAlert extends Notification implements ShouldQueue
{
    use Queueable;

    protected $batches;
    protected $alert;

    /**
     * Create a new notification instance.
     */
    public function __construct(Collection $batches, InventoryAlert $alert)
    {
        $this->batches = $batches;
        $this->alert = $alert;
    }

    /**
     * Get the notification's delivery channels.
     *
     * @return array<int, string>
     */
    public function via(object $notifiable): array
    {
        return ['mail', 'database'];
    }

    /**
     * Get the mail representation of the notification.
     */
    public function toMail(object $notifiable): MailMessage
    {
        $mailMessage = (new MailMessage)
            ->subject('Inventory Expiry Alert - Action Required')
            ->greeting('Hello ' . $notifiable->name . ',')
            ->line('The following inventory batches will expire soon and require your attention:')
            ->line('');

        foreach ($this->batches as $batch) {
            $daysRemaining = now()->diffInDays($batch->expiry_date);
            $mailMessage->line("• {$batch->item->name} (Batch: {$batch->batch_number}) - Expires in {$daysRemaining} days ({$batch->expiry_date->format('Y-m-d')}), Quantity: {$batch->quantity}");
        }

        return $mailMessage
            ->line('')
            ->line("These batches will expire within the next {$this->alert->expiry_threshold_days} days.")
            ->action('View Expiring Items', route('inventory.alerts.expiry'))
            ->line('Please take appropriate action to manage these expiring items.');
    }

    /**
     * Get the array representation of the notification.
     *
     * @return array<string, mixed>
     */
    public function toArray(object $notifiable): array
    {
        return [
            'title' => 'Inventory Expiry Alert',
            'message' => count($this->batches) . ' batches will expire soon',
            'action_url' => route('inventory.alerts.expiry'),
            'batches' => $this->batches->map(function ($batch) {
                return [
                    'id' => $batch->id,
                    'item_name' => $batch->item->name,
                    'batch_number' => $batch->batch_number,
                    'expiry_date' => $batch->expiry_date->format('Y-m-d'),
                    'days_remaining' => now()->diffInDays($batch->expiry_date),
                    'quantity' => $batch->quantity,
                ];
            }),
        ];
    }
}
