<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class SalesReturn extends Model
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'company_id',
        'customer_id',
        'invoice_id',
        'warehouse_id',
        'created_by',
        'return_number',
        'return_date',
        'subtotal',
        'tax_amount',
        'discount_amount',
        'total_amount',
        'status',
        'notes',
        'internal_notes',
        'approved_by',
        'approved_at',
    ];

    protected $casts = [
        'return_date' => 'date',
        'approved_at' => 'datetime',
    ];

    /**
     * تطبيق عزل الشركات
     */
    protected static function booted()
    {
        static::addGlobalScope('company', function ($query) {
            if (auth()->check() && !auth()->user()->isSystemAdmin()) {
                $query->where('company_id', auth()->user()->company_id);
            }
        });
    }

    /**
     * علاقة مع الشركة
     */
    public function company()
    {
        return $this->belongsTo(Company::class);
    }

    /**
     * علاقة مع العميل
     */
    public function customer()
    {
        return $this->belongsTo(Customer::class);
    }

    /**
     * علاقة مع الفاتورة الأصلية
     */
    public function invoice()
    {
        return $this->belongsTo(Invoice::class);
    }

    /**
     * علاقة مع المستودع
     */
    public function warehouse()
    {
        return $this->belongsTo(Warehouse::class);
    }

    /**
     * علاقة مع المستخدم الذي أنشأ المرتجع
     */
    public function creator()
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    /**
     * علاقة مع المستخدم الذي اعتمد المرتجع
     */
    public function approver()
    {
        return $this->belongsTo(User::class, 'approved_by');
    }

    /**
     * علاقة مع بنود المرتجع
     */
    public function items()
    {
        return $this->hasMany(SalesReturnItem::class);
    }

    /**
     * علاقة مع القيود المحاسبية
     */
    public function journalEntries()
    {
        return $this->morphMany(JournalEntry::class, 'reference');
    }

    /**
     * علاقة مع حركات المخزون
     */
    public function inventoryTransactions()
    {
        return $this->morphMany(InventoryTransaction::class, 'reference');
    }
}
