<?php

/**
 * Created by Reliese Model.
 */

namespace App\Models;

use Carbon\Carbon;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

/**
 * Class Role
 * 
 * @property int $id
 * @property int|null $company_id
 * @property string $name
 * @property string $slug
 * @property string|null $description
 * @property int $level
 * @property string|null $path
 * @property bool $is_active
 * @property bool $is_system_role
 * @property bool $is_default
 * @property int $order
 * @property int|null $parent_id
 * @property Carbon|null $created_at
 * @property Carbon|null $updated_at
 * @property string|null $deleted_at
 * 
 * @property Company|null $company
 * @property Role|null $role
 * @property Collection|Permission[] $permissions
 * @property Collection|User[] $users
 * @property Collection|Role[] $roles
 *
 * @package App\Models
 */
class Role extends Model
{
	use SoftDeletes;
	protected $table = 'roles';

	protected $casts = [
		'company_id' => 'int',
		'level' => 'int',
		'is_active' => 'bool',
		'is_system_role' => 'bool',
		'is_default' => 'bool',
		'order' => 'int',
		'parent_id' => 'int'
	];

	protected $fillable = [
		'company_id',
		'name',
		'slug',
		'description',
		'level',
		'path',
		'is_active',
		'is_system_role',
		'is_default',
		'order',
		'parent_id'
	];

	public function company()
	{
		return $this->belongsTo(Company::class);
	}

	public function role()
	{
		return $this->belongsTo(Role::class, 'parent_id');
	}

	public function permissions()
	{
		return $this->belongsToMany(Permission::class)
			->withPivot('id')
			->withTimestamps();
	}

	public function users()
	{
		return $this->belongsToMany(User::class)
			->withPivot('id')
			->withTimestamps();
	}

	public function roles()
	{
		return $this->hasMany(Role::class, 'parent_id');
	}

	/**
	 * الحصول على الأدوار الفرعية
	 *
	 * @return \Illuminate\Database\Eloquent\Relations\HasMany
	 */
	public function children()
	{
		return $this->hasMany(Role::class, 'parent_id');
	}

	/**
	 * تحديث مسار الدور
	 *
	 * @return void
	 */
	public function updatePath()
	{
		if ($this->parent_id) {
			$parent = Role::find($this->parent_id);
			if ($parent) {
				$this->path = $parent->path ? $parent->path . '.' . $this->id : $this->id;
			} else {
				$this->path = (string) $this->id;
			}
		} else {
			$this->path = (string) $this->id;
		}
		$this->save();

		// تحديث مسارات الأدوار الفرعية
		foreach ($this->children as $child) {
			$child->updatePath();
		}
	}

	/**
	 * تحديث مستوى الدور
	 *
	 * @return void
	 */
	public function updateLevel()
	{
		if ($this->parent_id) {
			$parent = Role::find($this->parent_id);
			if ($parent) {
				$this->level = $parent->level + 1;
			} else {
				$this->level = 0;
			}
		} else {
			$this->level = 0;
		}
		$this->save();

		// تحديث مستويات الأدوار الفرعية
		foreach ($this->children as $child) {
			$child->updateLevel();
		}
	}
}
