<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class PurchaseReturn extends Model
{
    use HasFactory, SoftDeletes;

    /**
     * الحقول القابلة للتعبئة الجماعية
     *
     * @var array
     */
    protected $fillable = [
        'return_number',
        'company_id',
        'supplier_id',
        'warehouse_id',
        'purchase_invoice_id',
        'created_by',
        'return_date',
        'subtotal',
        'tax_amount',
        'discount_amount',
        'total_amount',
        'notes',
        'attachment',
    ];

    /**
     * الحقول التي يجب تحويلها إلى أنواع محددة
     *
     * @var array
     */
    protected $casts = [
        'return_date' => 'date',
    ];

    /**
     * علاقة مع الشركة
     */
    public function company()
    {
        return $this->belongsTo(Company::class);
    }

    /**
     * علاقة مع المورد
     */
    public function supplier()
    {
        return $this->belongsTo(Supplier::class);
    }

    /**
     * علاقة مع المستودع
     */
    public function warehouse()
    {
        return $this->belongsTo(Warehouse::class);
    }

    /**
     * علاقة مع فاتورة المشتريات الأصلية
     */
    public function purchaseInvoice()
    {
        return $this->belongsTo(PurchaseInvoice::class);
    }

    /**
     * علاقة مع المستخدم الذي أنشأ المرتجع
     */
    public function creator()
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    /**
     * علاقة مع عناصر مرتجع المشتريات
     */
    public function items()
    {
        return $this->hasMany(PurchaseReturnItem::class);
    }

    /**
     * علاقة مع القيد المحاسبي
     */
    public function journalEntry()
    {
        return $this->morphOne(JournalEntry::class, 'reference');
    }

    /**
     * توليد رقم مرتجع فريد
     *
     * @param int $companyId معرف الشركة
     * @return string رقم المرتجع الجديد
     */
    public static function generateReturnNumber($companyId)
    {
        // البحث عن آخر مرتجع للشركة
        $lastReturn = self::where('company_id', $companyId)
            ->orderBy('id', 'desc')
            ->first();

        if (!$lastReturn) {
            // إذا لم يكن هناك مرتجعات سابقة، نبدأ من 1
            return 'PR-' . str_pad(1, 6, '0', STR_PAD_LEFT);
        }

        // استخراج الرقم من آخر مرتجع
        if (preg_match('/PR-(\d+)/', $lastReturn->return_number, $matches)) {
            $nextNumber = (int)$matches[1] + 1;
        } else {
            // إذا كان تنسيق الرقم غير متوقع، نستخدم معرف المرتجع + 1
            $nextNumber = $lastReturn->id + 1;
        }

        // تنسيق الرقم الجديد
        return 'PR-' . str_pad($nextNumber, 6, '0', STR_PAD_LEFT);
    }
}
