<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class PurchaseInvoice extends Model
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'invoice_number',
        'company_id',
        'purchase_order_id',
        'supplier_id',
        'warehouse_id',
        'created_by',
        'invoice_date',
        'due_date',
        'subtotal',
        'tax_amount',
        'discount_amount',
        'total_amount',
        'payment_status',
        'notes',
        'supplier_invoice_number',
        'attachment',
    ];

    protected $dates = [
        'invoice_date',
        'due_date',
        'created_at',
        'updated_at',
        'deleted_at',
    ];

    /**
     * العلاقة مع الشركة
     */
    public function company()
    {
        return $this->belongsTo(Company::class);
    }

    /**
     * العلاقة مع المورد
     */
    public function supplier()
    {
        return $this->belongsTo(Supplier::class);
    }

    /**
     * العلاقة مع المستودع
     */
    public function warehouse()
    {
        return $this->belongsTo(Warehouse::class);
    }

    /**
     * العلاقة مع المستخدم الذي أنشأ الفاتورة
     */
    public function creator()
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    /**
     * العلاقة مع أمر الشراء
     */
    public function purchaseOrder()
    {
        return $this->belongsTo(PurchaseOrder::class);
    }

    /**
     * العلاقة مع عناصر الفاتورة
     */
    public function items()
    {
        return $this->hasMany(PurchaseInvoiceItem::class);
    }

    /**
     * العلاقة مع مدفوعات الفاتورة
     */
    public function payments()
    {
        return $this->hasMany(PurchaseInvoicePayment::class);
    }

    /**
     * حساب المبلغ المدفوع
     */
    public function getPaidAmountAttribute()
    {
        return $this->payments()->sum('amount');
    }

    /**
     * حساب المبلغ المتبقي
     */
    public function getRemainingAmountAttribute()
    {
        return $this->total_amount - $this->paid_amount;
    }

    /**
     * تحديث حالة الدفع بناءً على المدفوعات
     */
    public function updatePaymentStatus()
    {
        $paidAmount = $this->paid_amount;

        if ($paidAmount <= 0) {
            $this->payment_status = 'unpaid';
        } elseif ($paidAmount < $this->total_amount) {
            $this->payment_status = 'partially_paid';
        } else {
            $this->payment_status = 'paid';
        }

        $this->save();

        return $this;
    }
}
