<?php

/**
 * Created by Reliese Model.
 */

namespace App\Models;

use Carbon\Carbon;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Model;

/**
 * Class ProjectPhase
 * 
 * @property int $id
 * @property int $installation_project_id
 * @property string $name
 * @property string|null $description
 * @property Carbon $start_date
 * @property Carbon $end_date
 * @property int $order_number
 * @property float $budget
 * @property float $actual_cost
 * @property string $status
 * @property int $completion_percentage
 * @property int|null $responsible_user_id
 * @property string|null $notes
 * @property Carbon|null $created_at
 * @property Carbon|null $updated_at
 * 
 * @property InstallationProject $installation_project
 * @property User|null $user
 * @property Collection|ProjectChecklist[] $project_checklists
 * @property Collection|ProjectExpense[] $project_expenses
 * @property Collection|ProjectInventoryRequest[] $project_inventory_requests
 * @property Collection|ProjectIssue[] $project_issues
 * @property Collection|ProjectTask[] $project_tasks
 *
 * @package App\Models
 */
class ProjectPhase extends Model
{
	protected $table = 'project_phases';

	protected $casts = [
		'installation_project_id' => 'int',
		'start_date' => 'datetime',
		'end_date' => 'datetime',
		'order_number' => 'int',
		'budget' => 'float',
		'actual_cost' => 'float',
		'completion_percentage' => 'int',
		'responsible_user_id' => 'int'
	];

	protected $fillable = [
		'installation_project_id',
		'name',
		'description',
		'start_date',
		'end_date',
		'order_number',
		'budget',
		'actual_cost',
		'status',
		'completion_percentage',
		'responsible_user_id',
		'notes'
	];

	public function installation_project()
	{
		return $this->belongsTo(InstallationProject::class);
	}

	public function user()
	{
		return $this->belongsTo(User::class, 'responsible_user_id');
	}

	public function project_checklists()
	{
		return $this->hasMany(ProjectChecklist::class);
	}

	public function project_expenses()
	{
		return $this->hasMany(ProjectExpense::class);
	}

	public function project_inventory_requests()
	{
		return $this->hasMany(ProjectInventoryRequest::class);
	}

	public function project_issues()
	{
		return $this->hasMany(ProjectIssue::class);
	}

	public function project_tasks()
	{
		return $this->hasMany(ProjectTask::class);
	}

	// العلاقات المختصرة للاستخدام في المتحكم
	public function tasks()
	{
		return $this->hasMany(ProjectTask::class);
	}

	public function installationProject()
	{
		return $this->belongsTo(InstallationProject::class, 'installation_project_id');
	}

	public function responsibleUser()
	{
		return $this->belongsTo(User::class, 'responsible_user_id');
	}

	public function expenses()
	{
		return $this->hasMany(ProjectExpense::class);
	}

	public function issues()
	{
		return $this->hasMany(ProjectIssue::class);
	}

	public function checklists()
	{
		return $this->hasMany(ProjectChecklist::class);
	}

	/**
	 * تحديث نسبة الإنجاز للمرحلة بناءً على المهام
	 *
	 * @return void
	 */
	public function updateProgress()
	{
		// إذا لم يكن هناك مهام، لا تغير نسبة الإنجاز
		$tasks = $this->tasks;
		if ($tasks->isEmpty()) {
			return;
		}

		// حساب متوسط نسبة الإكمال لجميع المهام
		$totalProgress = $tasks->sum('completion_percentage');
		$averageProgress = round($totalProgress / $tasks->count());

		// تحديث نسبة الإنجاز للمرحلة
		$this->update([
			'completion_percentage' => $averageProgress,
			// تحديث الحالة بناءً على نسبة الإنجاز
			'status' => $this->determineStatus($averageProgress)
		]);

		// لا نحتاج لتحديث نسبة الإنجاز للمشروع لأن الجدول لا يحتوي على هذا الحقل
		// يمكن استخدام الدالة calculateProgress() للحصول على نسبة الإنجاز عند الحاجة
	}

	/**
	 * تحديد حالة المرحلة بناءً على نسبة الإنجاز
	 *
	 * @param int $progress
	 * @return string
	 */
	private function determineStatus($progress)
	{
		if ($progress == 100) {
			return 'completed';
		} elseif ($progress > 0) {
			return 'in_progress';
		} else {
			return 'not_started';
		}
	}
}
