<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class JournalTemplate extends Model
{
    use HasFactory, SoftDeletes;

    /**
     * الحقول القابلة للتعبئة الجماعية
     *
     * @var array
     */
    protected $fillable = [
        'company_id',
        'name',
        'code',
        'description',
        'is_active',
        'created_by'
    ];

    /**
     * الحقول التي يجب تحويلها إلى أنواع محددة
     *
     * @var array
     */
    protected $casts = [
        'is_active' => 'boolean'
    ];

    /**
     * علاقة مع الشركة
     */
    public function company()
    {
        return $this->belongsTo(Company::class);
    }

    /**
     * علاقة مع المستخدم الذي أنشأ القالب
     */
    public function creator()
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    /**
     * علاقة مع بنود القالب
     */
    public function items()
    {
        return $this->hasMany(JournalTemplateItem::class);
    }

    /**
     * نطاق للقوالب النشطة
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    /**
     * الحصول على إجمالي النسب المئوية للبنود المدينة
     */
    public function getTotalDebitPercentageAttribute()
    {
        return $this->items()->where('type', 'debit')->sum('percentage');
    }

    /**
     * الحصول على إجمالي النسب المئوية للبنود الدائنة
     */
    public function getTotalCreditPercentageAttribute()
    {
        return $this->items()->where('type', 'credit')->sum('percentage');
    }

    /**
     * الحصول على إجمالي المبالغ الثابتة للبنود المدينة
     */
    public function getTotalDebitFixedAmountAttribute()
    {
        return $this->items()->where('type', 'debit')->sum('fixed_amount');
    }

    /**
     * الحصول على إجمالي المبالغ الثابتة للبنود الدائنة
     */
    public function getTotalCreditFixedAmountAttribute()
    {
        return $this->items()->where('type', 'credit')->sum('fixed_amount');
    }

    /**
     * التحقق من توازن النسب المئوية
     */
    public function isPercentageBalanced()
    {
        return $this->total_debit_percentage == $this->total_credit_percentage;
    }

    /**
     * التحقق من توازن المبالغ الثابتة
     */
    public function isFixedAmountBalanced()
    {
        return $this->total_debit_fixed_amount == $this->total_credit_fixed_amount;
    }

    /**
     * التحقق من صحة القالب
     */
    public function isValid()
    {
        // التحقق من وجود بنود
        if ($this->items->isEmpty()) {
            return false;
        }

        // التحقق من توازن النسب المئوية أو المبالغ الثابتة
        $hasPercentage = $this->items()->whereNotNull('percentage')->exists();
        $hasFixedAmount = $this->items()->whereNotNull('fixed_amount')->exists();

        if ($hasPercentage && !$this->isPercentageBalanced()) {
            return false;
        }

        if ($hasFixedAmount && !$this->isFixedAmountBalanced()) {
            return false;
        }

        return true;
    }
}
