<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class JournalEntryItem extends Model
{
    use HasFactory;

    /**
     * الحقول القابلة للتعبئة الجماعية
     *
     * @var array
     */
    protected $fillable = [
        'journal_entry_id',
        'account_id',
        'debit',
        'credit',
        'description',
        'reference_type',
        'reference_id'
    ];

    /**
     * الحقول التي يجب تحويلها إلى أنواع محددة
     *
     * @var array
     */
    protected $casts = [
        'debit' => 'decimal:2',
        'credit' => 'decimal:2'
    ];

    /**
     * علاقة مع القيد المحاسبي
     */
    public function journalEntry()
    {
        return $this->belongsTo(JournalEntry::class);
    }

    /**
     * علاقة مع الحساب
     */
    public function account()
    {
        return $this->belongsTo(ChartOfAccount::class, 'account_id');
    }

    /**
     * Alias for account relationship to maintain compatibility
     */
    public function chartOfAccount()
    {
        return $this->belongsTo(ChartOfAccount::class, 'account_id');
    }

    /**
     * علاقة مع المرجع (مورفيك)
     */
    public function reference()
    {
        return $this->morphTo();
    }

    /**
     * الحصول على نوع البند (مدين أو دائن)
     */
    public function getTypeAttribute()
    {
        if ($this->debit > 0) {
            return 'debit';
        } elseif ($this->credit > 0) {
            return 'credit';
        }

        return null;
    }

    /**
     * الحصول على قيمة البند (المدين أو الدائن)
     */
    public function getAmountAttribute()
    {
        return $this->debit > 0 ? $this->debit : $this->credit;
    }

    /**
     * الحصول على اسم نوع البند بالعربية
     */
    public function getTypeNameAttribute()
    {
        return $this->type == 'debit' ? 'مدين' : 'دائن';
    }

    /**
     * نطاق للبنود المدينة
     */
    public function scopeDebit($query)
    {
        return $query->where('debit', '>', 0);
    }

    /**
     * نطاق للبنود الدائنة
     */
    public function scopeCredit($query)
    {
        return $query->where('credit', '>', 0);
    }

    /**
     * نطاق للبنود المرتبطة بنوع حساب معين
     */
    public function scopeAccountType($query, $accountType)
    {
        return $query->whereHas('account', function ($q) use ($accountType) {
            $q->where('account_type', $accountType);
        });
    }
}
