<?php

/**
 * Created by Reliese Model.
 */

namespace App\Models;

use Carbon\Carbon;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Model;

/**
 * Class InventoryItem
 * 
 * @property int $id
 * @property int $company_id
 * @property int|null $category_id
 * @property string $name
 * @property string $sku
 * @property string|null $barcode
 * @property string|null $description
 * @property string|null $unit
 * @property int|null $base_unit_id
 * @property float $purchase_price
 * @property float $selling_price
 * @property float $min_stock_level
 * @property float $max_stock_level
 * @property bool $track_serial_numbers
 * @property bool $track_batches
 * @property bool $is_active
 * @property string|null $image
 * @property string|null $technical_specs
 * @property Carbon|null $created_at
 * @property Carbon|null $updated_at
 * 
 * @property InventoryCategory|null $inventory_category
 * @property InventoryUnit|null $base_unit
 * @property Company $company
 * @property Collection|AutoPurchaseRule[] $auto_purchase_rules
 * @property Collection|InventoryBatch[] $inventory_batches
 * @property Collection|InventorySerial[] $inventory_serials
 * @property Collection|InventoryStock[] $inventory_stocks
 * @property Collection|InventoryTransaction[] $inventory_transactions
 * @property Collection|InventoryValuation[] $inventory_valuations
 * @property Collection|PurchaseOrderItem[] $purchase_order_items
 * @property Collection|InventoryItemUnit[] $inventory_item_units
 *
 * @package App\Models
 */
class InventoryItem extends Model
{
	protected $table = 'inventory_items';

	protected $casts = [
		'company_id' => 'int',
		'category_id' => 'int',
		'base_unit_id' => 'int',
		'purchase_price' => 'float',
		'selling_price' => 'float',
		'min_stock_level' => 'float',
		'max_stock_level' => 'float',
		'track_serial_numbers' => 'bool',
		'track_batches' => 'bool',
		'is_active' => 'bool'
	];

	protected $fillable = [
		'company_id',
		'category_id',
		'name',
		'sku',
		'barcode',
		'description',
		'unit',
		'base_unit_id',
		'purchase_price',
		'selling_price',
		'min_stock_level',
		'max_stock_level',
		'track_serial_numbers',
		'track_batches',
		'is_active',
		'image',
		'technical_specs'
	];

	public function inventory_category()
	{
		return $this->belongsTo(InventoryCategory::class, 'category_id');
	}

	public function category()
	{
		return $this->belongsTo(InventoryCategory::class, 'category_id');
	}

	public function base_unit()
	{
		return $this->belongsTo(InventoryUnit::class, 'base_unit_id');
	}

	public function company()
	{
		return $this->belongsTo(Company::class);
	}

	public function inventory_item_units()
	{
		return $this->hasMany(InventoryItemUnit::class, 'item_id');
	}

	public function auto_purchase_rules()
	{
		return $this->hasMany(AutoPurchaseRule::class, 'item_id');
	}

	public function inventory_batches()
	{
		return $this->hasMany(InventoryBatch::class, 'item_id');
	}

	public function inventory_serials()
	{
		return $this->hasMany(InventorySerial::class, 'item_id');
	}

	public function inventory_stocks()
	{
		return $this->hasMany(InventoryStock::class, 'item_id');
	}

	public function stocks()
	{
		return $this->hasMany(InventoryStock::class, 'item_id');
	}

	public function inventory_transactions()
	{
		return $this->hasMany(InventoryTransaction::class, 'item_id');
	}

	public function inventory_valuations()
	{
		return $this->hasMany(InventoryValuation::class, 'item_id');
	}

	public function purchase_order_items()
	{
		return $this->hasMany(PurchaseOrderItem::class, 'item_id');
	}

	/**
	 * Determine if the item is low on stock
	 *
	 * @return bool
	 */
	public function getIsLowStockAttribute()
	{
		$totalStock = $this->stocks()->sum('quantity');
		return $totalStock <= $this->min_stock_level;
	}
}
