<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class InventoryCount extends Model
{
    use HasFactory;

    protected $fillable = [
        'company_id',
        'warehouse_id',
        'count_number',
        'count_date',
        'count_type',
        'status',
        'notes',
        'created_by',
        'completed_by',
        'completed_at',
        'approved_by',
        'approved_at',
        'cancelled_by',
        'cancelled_at',
    ];

    protected $casts = [
        'count_date' => 'date',
        'completed_at' => 'datetime',
        'approved_at' => 'datetime',
        'cancelled_at' => 'datetime',
    ];

    public function company()
    {
        return $this->belongsTo(Company::class);
    }

    public function warehouse()
    {
        return $this->belongsTo(Warehouse::class);
    }

    public function createdBy()
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    public function completedBy()
    {
        return $this->belongsTo(User::class, 'completed_by');
    }

    public function approvedBy()
    {
        return $this->belongsTo(User::class, 'approved_by');
    }

    public function cancelledBy()
    {
        return $this->belongsTo(User::class, 'cancelled_by');
    }

    public function items()
    {
        return $this->hasMany(InventoryCountItem::class);
    }

    /**
     * الحصول على عدد العناصر التي تم عدها
     */
    public function getCountedItemsCountAttribute()
    {
        return $this->items()->whereNotNull('counted_at')->count();
    }

    /**
     * الحصول على إجمالي عدد العناصر
     */
    public function getTotalItemsCountAttribute()
    {
        return $this->items()->count();
    }

    /**
     * الحصول على عدد العناصر المطابقة
     */
    public function getMatchingItemsCountAttribute()
    {
        return $this->items()->where('status', 'match')->count();
    }

    /**
     * الحصول على عدد العناصر الزائدة
     */
    public function getSurplusItemsCountAttribute()
    {
        return $this->items()->where('status', 'surplus')->count();
    }

    /**
     * الحصول على عدد العناصر الناقصة
     */
    public function getShortageItemsCountAttribute()
    {
        return $this->items()->where('status', 'shortage')->count();
    }

    /**
     * الحصول على نسبة الدقة
     */
    public function getAccuracyPercentageAttribute()
    {
        $totalItems = $this->total_items_count;
        if ($totalItems == 0) {
            return 0;
        }

        $matchingItems = $this->matching_items_count;
        return round(($matchingItems / $totalItems) * 100, 2);
    }

    /**
     * الحصول على حالة الجلسة بالعربية
     */
    public function getStatusTextAttribute()
    {
        $statuses = [
            'draft' => 'مسودة',
            'in_progress' => 'قيد التنفيذ',
            'completed' => 'مكتمل',
            'approved' => 'معتمد',
            'cancelled' => 'ملغي',
        ];

        return $statuses[$this->status] ?? $this->status;
    }

    /**
     * الحصول على نوع الجرد بالعربية
     */
    public function getCountTypeTextAttribute()
    {
        $types = [
            'full' => 'جرد كامل',
            'partial' => 'جرد جزئي',
            'cycle' => 'جرد دوري',
        ];

        return $types[$this->count_type] ?? $this->count_type;
    }
}
