<?php

/**
 * Created by Reliese Model.
 */

namespace App\Models;

use Carbon\Carbon;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

/**
 * Class InstallationProject
 * 
 * @property int $id
 * @property int $company_id
 * @property int|null $branch_id
 * @property int $customer_id
 * @property int $location_id
 * @property string $project_number
 * @property string $title
 * @property string|null $description
 * @property Carbon $start_date
 * @property Carbon $expected_completion_date
 * @property Carbon|null $actual_completion_date
 * @property float $total_budget
 * @property float $total_cost
 * @property string $status
 * @property string $payment_status
 * @property int $project_manager_id
 * @property string|null $notes
 * @property Carbon|null $created_at
 * @property Carbon|null $updated_at
 * @property string|null $deleted_at
 * 
 * @property Branch|null $branch
 * @property Company $company
 * @property Customer $customer
 * @property Location $location
 * @property User $user
 * @property Collection|ProjectChecklist[] $project_checklists
 * @property Collection|ProjectDocument[] $project_documents
 * @property Collection|ProjectElevator[] $project_elevators
 * @property Collection|ProjectExpense[] $project_expenses
 * @property Collection|ProjectInventoryRequest[] $project_inventory_requests
 * @property Collection|ProjectIssue[] $project_issues
 * @property Collection|ProjectMilestone[] $project_milestones
 * @property Collection|ProjectPayment[] $project_payments
 * @property Collection|ProjectPhase[] $project_phases
 * @property Collection|ProjectReport[] $project_reports
 * @property Collection|ProjectResource[] $project_resources
 * @property Collection|ProjectRisk[] $project_risks
 * @property Collection|ProjectTeamMember[] $project_team_members
 *
 * @package App\Models
 */
class InstallationProject extends Model
{
	use SoftDeletes;
	protected $table = 'installation_projects';

	protected $casts = [
		'company_id' => 'int',
		'branch_id' => 'int',
		'customer_id' => 'int',
		'location_id' => 'int',
		'start_date' => 'datetime',
		'expected_completion_date' => 'datetime',
		'actual_completion_date' => 'datetime',
		'total_budget' => 'float',
		'total_cost' => 'float',
		'project_manager_id' => 'int'
	];

	protected $fillable = [
		'company_id',
		'branch_id',
		'customer_id',
		'location_id',
		'project_number',
		'title',
		'description',
		'start_date',
		'expected_completion_date',
		'actual_completion_date',
		'total_budget',
		'total_cost',
		'status',
		'payment_status',
		'project_manager_id',
		'notes'
	];

	public function branch()
	{
		return $this->belongsTo(Branch::class);
	}

	public function company()
	{
		return $this->belongsTo(Company::class);
	}

	public function customer()
	{
		return $this->belongsTo(Customer::class);
	}

	public function location()
	{
		return $this->belongsTo(Location::class);
	}

	public function user()
	{
		return $this->belongsTo(User::class, 'project_manager_id');
	}

	public function projectManager()
	{
		return $this->belongsTo(User::class, 'project_manager_id');
	}

	public function project_checklists()
	{
		return $this->hasMany(ProjectChecklist::class);
	}

	public function project_documents()
	{
		return $this->hasMany(ProjectDocument::class);
	}

	public function project_elevators()
	{
		return $this->hasMany(ProjectElevator::class);
	}

	public function project_expenses()
	{
		return $this->hasMany(ProjectExpense::class);
	}

	public function project_inventory_requests()
	{
		return $this->hasMany(ProjectInventoryRequest::class);
	}

	public function project_issues()
	{
		return $this->hasMany(ProjectIssue::class);
	}

	public function project_milestones()
	{
		return $this->hasMany(ProjectMilestone::class);
	}

	public function project_payments()
	{
		return $this->hasMany(ProjectPayment::class);
	}

	public function project_phases()
	{
		return $this->hasMany(ProjectPhase::class);
	}

	public function project_reports()
	{
		return $this->hasMany(ProjectReport::class);
	}

	public function project_resources()
	{
		return $this->hasMany(ProjectResource::class);
	}

	public function project_risks()
	{
		return $this->hasMany(ProjectRisk::class);
	}

	public function project_team_members()
	{
		return $this->hasMany(ProjectTeamMember::class);
	}

	// العلاقات المختصرة للاستخدام في المتحكم

	public function elevators()
	{
		return $this->hasMany(ProjectElevator::class);
	}

	public function phases()
	{
		return $this->hasMany(ProjectPhase::class);
	}

	public function teamMembers()
	{
		return $this->hasMany(ProjectTeamMember::class);
	}

	public function milestones()
	{
		return $this->hasMany(ProjectMilestone::class);
	}

	public function documents()
	{
		return $this->hasMany(ProjectDocument::class);
	}

	public function expenses()
	{
		return $this->hasMany(ProjectExpense::class);
	}

	public function payments()
	{
		return $this->hasMany(ProjectPayment::class);
	}

	public function issues()
	{
		return $this->hasMany(ProjectIssue::class);
	}

	/**
	 * حساب نسبة تقدم المشروع
	 *
	 * @return float
	 */
	public function calculateProgress()
	{
		// إذا كان المشروع مكتمل، أعد 100%
		if ($this->status === 'completed') {
			return 100;
		}

		// إذا لم يكن هناك مراحل، أعد 0%
		$phases = $this->phases;
		if ($phases->isEmpty()) {
			return 0;
		}

		// حساب متوسط نسبة الإكمال لجميع المراحل
		$totalProgress = $phases->sum('completion_percentage');
		return round($totalProgress / $phases->count(), 2);
	}

	/**
	 * حساب إجمالي تكلفة المشروع
	 *
	 * @return float
	 */
	public function calculateTotalCost()
	{
		// إجمالي المصروفات
		$expensesTotal = $this->expenses()->sum('amount');

		// إضافة أي تكاليف أخرى
		return $expensesTotal + $this->total_cost;
	}

	/**
	 * حساب إجمالي المدفوعات
	 *
	 * @return float
	 */
	public function calculateTotalPayments()
	{
		return $this->payments()->sum('amount');
	}

	/**
	 * حساب الميزانية المتبقية
	 *
	 * @return float
	 */
	public function getRemainingBudget()
	{
		return $this->total_budget - $this->calculateTotalCost();
	}

	/**
	 * حساب الأيام المتبقية للمشروع
	 *
	 * @return int
	 */
	public function getDaysRemaining()
	{
		// إذا كان المشروع مكتمل، أعد 0
		if ($this->status === 'completed') {
			return 0;
		}

		$today = now();
		$completionDate = $this->expected_completion_date;

		// إذا كان تاريخ الإكمال المتوقع قد مر، أعد عدد سالب
		if ($today > $completionDate) {
			return -$today->diffInDays($completionDate);
		}

		return $today->diffInDays($completionDate);
	}

	/**
	 * التحقق مما إذا كان المشروع متأخرًا
	 *
	 * @return bool
	 */
	public function isOverdue()
	{
		// إذا كان المشروع مكتمل، فهو ليس متأخرًا
		if ($this->status === 'completed') {
			return false;
		}

		return now() > $this->expected_completion_date;
	}
}
