<?php

namespace App\Models\HRM;

use App\Models\Company;
use App\Models\User;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Loan extends Model
{
    use HasFactory;

    protected $fillable = [
        'company_id',
        'employee_id',
        'loan_type',
        'amount',
        'interest_rate',
        'interest_amount',
        'total_amount',
        'installment_amount',
        'installment_count',
        'remaining_installments',
        'start_date',
        'end_date',
        'status',
        'approved_by',
        'approved_at',
        'description',
        'notes',
        'created_by',
        'updated_by'
    ];

    protected $casts = [
        'amount' => 'decimal:2',
        'interest_rate' => 'decimal:2',
        'interest_amount' => 'decimal:2',
        'total_amount' => 'decimal:2',
        'installment_amount' => 'decimal:2',
        'start_date' => 'date',
        'end_date' => 'date',
        'approved_at' => 'datetime',
    ];

    /**
     * أنواع القروض المتاحة
     */
    const LOAN_TYPES = [
        'personal' => 'قرض شخصي',
        'housing' => 'قرض سكني',
        'car' => 'قرض سيارة',
        'emergency' => 'قرض طارئ',
        'advance' => 'سلفة',
        'other' => 'أخرى'
    ];

    /**
     * حالات القرض
     */
    const STATUSES = [
        'pending' => 'قيد الانتظار',
        'approved' => 'معتمد',
        'rejected' => 'مرفوض',
        'active' => 'نشط',
        'completed' => 'مكتمل',
        'cancelled' => 'ملغي'
    ];

    /**
     * العلاقة مع الشركة
     */
    public function company()
    {
        return $this->belongsTo(Company::class);
    }

    /**
     * العلاقة مع الموظف
     */
    public function employee()
    {
        return $this->belongsTo(Employee::class);
    }

    /**
     * العلاقة مع المستخدم الذي وافق على القرض
     */
    public function approver()
    {
        return $this->belongsTo(User::class, 'approved_by');
    }

    /**
     * العلاقة مع المستخدم الذي أنشأ القرض
     */
    public function creator()
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    /**
     * العلاقة مع أقساط القرض
     */
    public function installments()
    {
        return $this->hasMany(LoanInstallment::class);
    }

    /**
     * الحصول على اسم نوع القرض
     */
    public function getLoanTypeNameAttribute()
    {
        return self::LOAN_TYPES[$this->loan_type] ?? $this->loan_type;
    }

    /**
     * الحصول على اسم حالة القرض
     */
    public function getStatusNameAttribute()
    {
        return self::STATUSES[$this->status] ?? $this->status;
    }

    /**
     * حساب المبلغ المتبقي من القرض
     */
    public function getRemainingAmountAttribute()
    {
        $paidAmount = $this->installments()->where('status', 'paid')->sum('amount');
        return $this->total_amount - $paidAmount;
    }

    /**
     * حساب نسبة السداد
     */
    public function getPaymentPercentageAttribute()
    {
        if ($this->total_amount <= 0) {
            return 0;
        }

        $paidAmount = $this->installments()->where('status', 'paid')->sum('amount');
        return round(($paidAmount / $this->total_amount) * 100, 2);
    }
}
