<?php

namespace App\Models\HRM;

use App\Models\Company;
use App\Models\User;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class EmployeeTask extends Model
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'company_id',
        'title',
        'description',
        'priority',
        'status',
        'start_date',
        'due_date',
        'completed_date',
        'assigned_by',
        'notes',
        'created_by',
        'updated_by'
    ];

    protected $casts = [
        'start_date' => 'date',
        'due_date' => 'date',
        'completed_date' => 'date',
    ];

    /**
     * العلاقة مع الشركة
     */
    public function company()
    {
        return $this->belongsTo(Company::class);
    }

    /**
     * العلاقة مع الموظفين المكلفين بالمهمة
     */
    public function assignees()
    {
        return $this->hasMany(EmployeeTaskAssignee::class, 'task_id');
    }

    /**
     * العلاقة مع المستخدم الذي أسند المهمة
     */
    public function assigner()
    {
        return $this->belongsTo(User::class, 'assigned_by');
    }

    /**
     * العلاقة مع المستخدم الذي أنشأ المهمة
     */
    public function creator()
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    /**
     * العلاقة مع المستخدم الذي قام بتحديث المهمة
     */
    public function updater()
    {
        return $this->belongsTo(User::class, 'updated_by');
    }

    /**
     * العلاقة مع تعليقات المهمة
     */
    public function comments()
    {
        return $this->hasMany(EmployeeTaskComment::class, 'task_id');
    }

    /**
     * العلاقة مع مرفقات المهمة
     */
    public function attachments()
    {
        return $this->hasMany(EmployeeTaskAttachment::class, 'task_id');
    }

    /**
     * التحقق مما إذا كانت المهمة متأخرة
     */
    public function getIsOverdueAttribute()
    {
        if ($this->due_date && $this->status != 'completed') {
            return now()->gt($this->due_date);
        }
        return false;
    }

    /**
     * حساب عدد الأيام المتبقية حتى الموعد النهائي
     */
    public function getDaysUntilDueAttribute()
    {
        if ($this->due_date) {
            if ($this->status == 'completed' || now()->gt($this->due_date)) {
                return 0;
            }
            return now()->diffInDays($this->due_date);
        }
        return null;
    }

    /**
     * حساب مدة المهمة بالأيام
     */
    public function getDurationDaysAttribute()
    {
        if ($this->start_date && $this->due_date) {
            return $this->start_date->diffInDays($this->due_date) + 1;
        }
        return 0;
    }
}
