<?php

namespace App\Models\HRM;

use App\Models\Branch;
use App\Models\Company;
use App\Models\User;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class Employee extends Model
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'company_id',
        'branch_id',
        'department_id',
        'job_title_id',
        'user_id',
        'employee_number',
        'first_name',
        'last_name',
        'first_name_en',
        'last_name_en',
        'national_id',
        'passport_number',
        'passport_expiry',
        'nationality',
        'birth_date',
        'gender',
        'marital_status',
        'phone',
        'emergency_contact',
        'address',
        'email',
        'join_date',
        'probation_end_date',
        'contract_end_date',
        'basic_salary',
        'housing_allowance',
        'transportation_allowance',
        'other_allowances',
        'bank_name',
        'bank_account',
        'iban',
        'education_level',
        'education_field',
        'skills',
        'notes',
        'status',
        'profile_picture',
        'created_by',
        'updated_by'
    ];

    protected $casts = [
        'skills' => 'array',
        'birth_date' => 'date',
        'join_date' => 'date',
        'probation_end_date' => 'date',
        'contract_end_date' => 'date',
        'passport_expiry' => 'date',
        'basic_salary' => 'decimal:2',
        'housing_allowance' => 'decimal:2',
        'transportation_allowance' => 'decimal:2',
        'other_allowances' => 'decimal:2',
    ];

    /**
     * الحصول على الاسم الكامل للموظف
     */
    public function getFullNameAttribute()
    {
        return $this->first_name . ' ' . $this->last_name;
    }

    /**
     * الحصول على الاسم الكامل للموظف بالإنجليزية
     */
    public function getFullNameEnAttribute()
    {
        if (!empty($this->first_name_en) && !empty($this->last_name_en)) {
            return $this->first_name_en . ' ' . $this->last_name_en;
        }
        return $this->first_name . ' ' . $this->last_name;
    }

    /**
     * الحصول على إجمالي الراتب
     */
    public function getTotalSalaryAttribute()
    {
        return $this->basic_salary + $this->housing_allowance + $this->transportation_allowance + $this->other_allowances;
    }

    /**
     * العلاقة مع الشركة
     */
    public function company()
    {
        return $this->belongsTo(Company::class);
    }

    /**
     * العلاقة مع الفرع
     */
    public function branch()
    {
        return $this->belongsTo(Branch::class);
    }

    /**
     * العلاقة مع القسم
     */
    public function department()
    {
        return $this->belongsTo(\App\Models\Department::class);
    }

    /**
     * العلاقة مع المسمى الوظيفي
     */
    public function jobTitle()
    {
        return $this->belongsTo(\App\Models\JobTitle::class);
    }

    /**
     * العلاقة مع المستخدم
     */
    public function user()
    {
        return $this->belongsTo(User::class);
    }

    /**
     * العلاقة مع الوثائق
     */
    public function documents()
    {
        return $this->hasMany(EmployeeDocument::class);
    }

    /**
     * العلاقة مع التقييمات
     */
    public function evaluations()
    {
        return $this->hasMany(EmployeeEvaluation::class);
    }

    /**
     * العلاقة مع سجلات الحضور
     */
    public function attendances()
    {
        return $this->hasMany(Attendance::class);
    }

    /**
     * العلاقة مع الإجازات
     */
    public function leaves()
    {
        return $this->hasMany(Leave::class);
    }

    /**
     * العلاقة مع عناصر الرواتب
     */
    public function payrollItems()
    {
        return $this->hasMany(PayrollItem::class);
    }

    /**
     * العلاقة مع المكافآت
     */
    public function bonuses()
    {
        return $this->hasMany(Bonus::class);
    }

    /**
     * العلاقة مع القروض
     */
    public function loans()
    {
        return $this->hasMany(Loan::class);
    }

    /**
     * العلاقة مع أقساط القروض
     */
    public function loanInstallments()
    {
        return $this->hasMany(LoanInstallment::class);
    }

    /**
     * حساب عدد أيام الإجازة المستخدمة في السنة الحالية
     */
    public function getUsedLeaveDaysAttribute()
    {
        $startOfYear = now()->startOfYear();
        $endOfYear = now()->endOfYear();

        return $this->leaves()
            ->where('status', 'approved')
            ->whereBetween('start_date', [$startOfYear, $endOfYear])
            ->sum('total_days');
    }

    /**
     * حساب عدد أيام الإجازة المتبقية في السنة الحالية
     * (افتراضياً 30 يوم في السنة)
     */
    public function getRemainingLeaveDaysAttribute()
    {
        // يمكن تعديل هذه القيمة حسب سياسة الشركة
        $annualLeaveEntitlement = 30;

        return $annualLeaveEntitlement - $this->used_leave_days;
    }

    /**
     * حساب عدد أيام الغياب في الشهر الحالي
     */
    public function getAbsenceDaysCurrentMonthAttribute()
    {
        $startOfMonth = now()->startOfMonth();
        $endOfMonth = now()->endOfMonth();
        $workingDays = 0;

        // حساب عدد أيام العمل في الشهر الحالي (باستثناء الجمعة والسبت)
        for ($date = clone $startOfMonth; $date <= $endOfMonth; $date->addDay()) {
            $dayOfWeek = $date->dayOfWeek;
            if ($dayOfWeek != 5 && $dayOfWeek != 6) { // 5 = الجمعة، 6 = السبت
                $workingDays++;
            }
        }

        // حساب عدد أيام الحضور
        $attendanceDays = $this->attendances()
            ->whereBetween('date', [$startOfMonth, $endOfMonth])
            ->where('status', 'present')
            ->count();

        // حساب عدد أيام الإجازات المعتمدة
        $leaveDays = $this->leaves()
            ->where('status', 'approved')
            ->where(function ($query) use ($startOfMonth, $endOfMonth) {
                $query->whereBetween('start_date', [$startOfMonth, $endOfMonth])
                    ->orWhereBetween('end_date', [$startOfMonth, $endOfMonth]);
            })
            ->sum('total_days');

        // عدد أيام الغياب = عدد أيام العمل - (عدد أيام الحضور + عدد أيام الإجازات المعتمدة)
        return max(0, $workingDays - ($attendanceDays + $leaveDays));
    }

    /**
     * حساب إجمالي ساعات العمل الإضافي في الشهر الحالي
     */
    public function getOvertimeHoursCurrentMonthAttribute()
    {
        $startOfMonth = now()->startOfMonth();
        $endOfMonth = now()->endOfMonth();

        return $this->attendances()
            ->whereBetween('date', [$startOfMonth, $endOfMonth])
            ->sum('overtime_minutes') / 60; // تحويل الدقائق إلى ساعات
    }

    /**
     * حساب إجمالي دقائق التأخير في الشهر الحالي
     */
    public function getLateMinutesCurrentMonthAttribute()
    {
        $startOfMonth = now()->startOfMonth();
        $endOfMonth = now()->endOfMonth();

        return $this->attendances()
            ->whereBetween('date', [$startOfMonth, $endOfMonth])
            ->sum('late_minutes');
    }
}
