<?php

namespace App\Models\HRM;

use App\Models\Company;
use App\Models\User;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class Certificate extends Model
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'company_id',
        'employee_id',
        'title',
        'certificate_type',
        'issuing_organization',
        'issue_date',
        'expiry_date',
        'certificate_number',
        'description',
        'file_path',
        'verification_link',
        'status',
        'notes',
        'created_by',
        'updated_by'
    ];

    protected $casts = [
        'issue_date' => 'date',
        'expiry_date' => 'date',
    ];

    /**
     * العلاقة مع الشركة
     */
    public function company()
    {
        return $this->belongsTo(Company::class);
    }

    /**
     * العلاقة مع الموظف
     */
    public function employee()
    {
        return $this->belongsTo(Employee::class);
    }

    /**
     * العلاقة مع المستخدم الذي أنشأ الشهادة
     */
    public function creator()
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    /**
     * العلاقة مع المستخدم الذي قام بتحديث الشهادة
     */
    public function updater()
    {
        return $this->belongsTo(User::class, 'updated_by');
    }

    /**
     * التحقق مما إذا كانت الشهادة منتهية الصلاحية
     */
    public function getIsExpiredAttribute()
    {
        if ($this->expiry_date) {
            return now()->gt($this->expiry_date);
        }
        return false;
    }

    /**
     * حساب عدد الأيام المتبقية حتى انتهاء الصلاحية
     */
    public function getDaysUntilExpiryAttribute()
    {
        if ($this->expiry_date) {
            if (now()->gt($this->expiry_date)) {
                return 0;
            }
            return now()->diffInDays($this->expiry_date);
        }
        return null;
    }
}
