<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class FiscalYear extends Model
{
    use HasFactory, SoftDeletes;

    /**
     * الحقول القابلة للتعبئة الجماعية
     *
     * @var array
     */
    protected $fillable = [
        'company_id',
        'name',
        'start_date',
        'end_date',
        'is_active',
        'is_closed'
    ];

    /**
     * الحقول التي يجب تحويلها إلى أنواع محددة
     *
     * @var array
     */
    protected $casts = [
        'start_date' => 'date',
        'end_date' => 'date',
        'is_active' => 'boolean',
        'is_closed' => 'boolean'
    ];

    /**
     * علاقة مع الشركة
     */
    public function company()
    {
        return $this->belongsTo(Company::class);
    }

    /**
     * علاقة مع الفترات المحاسبية
     */
    public function accountingPeriods()
    {
        return $this->hasMany(AccountingPeriod::class);
    }

    /**
     * علاقة مع القيود المحاسبية
     */
    public function journalEntries()
    {
        return $this->hasMany(JournalEntry::class);
    }

    /**
     * نطاق للسنوات المالية النشطة
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    /**
     * نطاق للسنوات المالية المفتوحة (غير مغلقة)
     */
    public function scopeOpen($query)
    {
        return $query->where('is_closed', false);
    }

    /**
     * نطاق للسنوات المالية المغلقة
     */
    public function scopeClosed($query)
    {
        return $query->where('is_closed', true);
    }

    /**
     * التحقق مما إذا كان التاريخ ضمن السنة المالية
     */
    public function isDateWithinYear($date)
    {
        $checkDate = $date instanceof \DateTime ? $date : new \DateTime($date);
        $startDate = new \DateTime($this->start_date);
        $endDate = new \DateTime($this->end_date);

        return $checkDate >= $startDate && $checkDate <= $endDate;
    }

    /**
     * إنشاء الفترات المحاسبية للسنة المالية
     * 
     * @param string $periodType شهري، ربع سنوي، نصف سنوي
     * @return void
     */
    public function createAccountingPeriods($periodType = 'monthly')
    {
        $startDate = new \DateTime($this->start_date);
        $endDate = new \DateTime($this->end_date);

        switch ($periodType) {
            case 'quarterly':
                $interval = new \DateInterval('P3M'); // كل 3 أشهر
                $periodCount = 4;
                break;
            case 'semi_annual':
                $interval = new \DateInterval('P6M'); // كل 6 أشهر
                $periodCount = 2;
                break;
            case 'monthly':
            default:
                $interval = new \DateInterval('P1M'); // كل شهر
                $periodCount = 12;
                break;
        }

        $currentStart = clone $startDate;
        $counter = 1;

        while ($currentStart < $endDate && $counter <= $periodCount) {
            $currentEnd = clone $currentStart;
            $currentEnd->add($interval);
            $currentEnd->modify('-1 day'); // نهاية الفترة هي اليوم السابق لبداية الفترة التالية

            // التأكد من أن نهاية الفترة لا تتجاوز نهاية السنة المالية
            if ($currentEnd > $endDate) {
                $currentEnd = clone $endDate;
            }

            // إنشاء الفترة المحاسبية
            $periodName = $this->generatePeriodName($periodType, $counter, $currentStart);

            AccountingPeriod::create([
                'fiscal_year_id' => $this->id,
                'company_id' => $this->company_id,
                'name' => $periodName,
                'start_date' => $currentStart->format('Y-m-d'),
                'end_date' => $currentEnd->format('Y-m-d'),
                'is_closed' => false
            ]);

            // الانتقال إلى الفترة التالية
            $currentStart = clone $currentEnd;
            $currentStart->modify('+1 day');
            $counter++;
        }
    }

    /**
     * توليد اسم الفترة المحاسبية
     */
    private function generatePeriodName($periodType, $counter, $startDate)
    {
        $yearName = $this->name;

        switch ($periodType) {
            case 'quarterly':
                return "الربع {$counter} - {$yearName}";
            case 'semi_annual':
                return $counter == 1 ? "النصف الأول - {$yearName}" : "النصف الثاني - {$yearName}";
            case 'monthly':
            default:
                $monthNames = [
                    'يناير',
                    'فبراير',
                    'مارس',
                    'إبريل',
                    'مايو',
                    'يونيو',
                    'يوليو',
                    'أغسطس',
                    'سبتمبر',
                    'أكتوبر',
                    'نوفمبر',
                    'ديسمبر'
                ];
                $monthIndex = (int)$startDate->format('n') - 1;
                return "{$monthNames[$monthIndex]} - {$yearName}";
        }
    }
}
