<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class EmployeeTaskAttachment extends Model
{
    use HasFactory;

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'task_id',
        'comment_id',
        'file_path',
        'file_name',
        'file_type',
        'file_size',
        'uploaded_by',
    ];

    /**
     * Get the task that owns the attachment.
     */
    public function task()
    {
        return $this->belongsTo(EmployeeTask::class, 'task_id');
    }

    /**
     * Get the comment that owns the attachment.
     */
    public function comment()
    {
        return $this->belongsTo(EmployeeTaskComment::class, 'comment_id');
    }

    /**
     * Get the user who uploaded the attachment.
     */
    public function uploadedBy()
    {
        return $this->belongsTo(User::class, 'uploaded_by');
    }

    /**
     * Get the formatted file size.
     */
    public function getFormattedFileSizeAttribute()
    {
        $bytes = $this->file_size;

        if ($bytes >= 1073741824) {
            return number_format($bytes / 1073741824, 2) . ' GB';
        } elseif ($bytes >= 1048576) {
            return number_format($bytes / 1048576, 2) . ' MB';
        } elseif ($bytes >= 1024) {
            return number_format($bytes / 1024, 2) . ' KB';
        } else {
            return $bytes . ' bytes';
        }
    }

    /**
     * Get the file icon based on file type.
     */
    public function getFileIconAttribute()
    {
        $fileType = $this->file_type;

        if (strpos($fileType, 'image/') === 0) {
            return 'fa-file-image';
        } elseif (strpos($fileType, 'application/pdf') === 0) {
            return 'fa-file-pdf';
        } elseif (strpos($fileType, 'application/msword') === 0 || strpos($fileType, 'application/vnd.openxmlformats-officedocument.wordprocessingml') === 0) {
            return 'fa-file-word';
        } elseif (strpos($fileType, 'application/vnd.ms-excel') === 0 || strpos($fileType, 'application/vnd.openxmlformats-officedocument.spreadsheetml') === 0) {
            return 'fa-file-excel';
        } elseif (strpos($fileType, 'application/vnd.ms-powerpoint') === 0 || strpos($fileType, 'application/vnd.openxmlformats-officedocument.presentationml') === 0) {
            return 'fa-file-powerpoint';
        } elseif (strpos($fileType, 'text/') === 0) {
            return 'fa-file-alt';
        } elseif (strpos($fileType, 'video/') === 0) {
            return 'fa-file-video';
        } elseif (strpos($fileType, 'audio/') === 0) {
            return 'fa-file-audio';
        } elseif (strpos($fileType, 'application/zip') === 0 || strpos($fileType, 'application/x-rar') === 0) {
            return 'fa-file-archive';
        } else {
            return 'fa-file';
        }
    }

    /**
     * Get the file color based on file type.
     */
    public function getFileColorAttribute()
    {
        $fileType = $this->file_type;

        if (strpos($fileType, 'image/') === 0) {
            return 'primary';
        } elseif (strpos($fileType, 'application/pdf') === 0) {
            return 'danger';
        } elseif (strpos($fileType, 'application/msword') === 0 || strpos($fileType, 'application/vnd.openxmlformats-officedocument.wordprocessingml') === 0) {
            return 'info';
        } elseif (strpos($fileType, 'application/vnd.ms-excel') === 0 || strpos($fileType, 'application/vnd.openxmlformats-officedocument.spreadsheetml') === 0) {
            return 'success';
        } elseif (strpos($fileType, 'application/vnd.ms-powerpoint') === 0 || strpos($fileType, 'application/vnd.openxmlformats-officedocument.presentationml') === 0) {
            return 'warning';
        } elseif (strpos($fileType, 'text/') === 0) {
            return 'secondary';
        } elseif (strpos($fileType, 'video/') === 0) {
            return 'danger';
        } elseif (strpos($fileType, 'audio/') === 0) {
            return 'info';
        } elseif (strpos($fileType, 'application/zip') === 0 || strpos($fileType, 'application/x-rar') === 0) {
            return 'warning';
        } else {
            return 'secondary';
        }
    }
}
