<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class EmployeeTask extends Model
{
    use HasFactory, SoftDeletes;

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'title',
        'description',
        'status',
        'priority',
        'progress',
        'due_date',
        'completed_at',
        'created_by',
        'updated_by',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array
     */
    protected $casts = [
        'due_date' => 'date',
        'completed_at' => 'datetime',
        'progress' => 'integer',
    ];

    /**
     * Get the assignees for the task.
     */
    public function assignees()
    {
        return $this->hasMany(EmployeeTaskAssignee::class, 'task_id');
    }

    /**
     * Get the comments for the task.
     */
    public function comments()
    {
        return $this->hasMany(EmployeeTaskComment::class, 'task_id')->orderBy('created_at', 'desc');
    }

    /**
     * Get the attachments for the task.
     */
    public function attachments()
    {
        return $this->hasMany(EmployeeTaskAttachment::class, 'task_id');
    }

    /**
     * Get the user who created the task.
     */
    public function createdBy()
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    /**
     * Get the user who last updated the task.
     */
    public function updatedBy()
    {
        return $this->belongsTo(User::class, 'updated_by');
    }

    /**
     * Get the status label.
     */
    public function getStatusLabelAttribute()
    {
        $labels = [
            'pending' => 'قيد الانتظار',
            'in_progress' => 'قيد التنفيذ',
            'completed' => 'مكتملة',
            'cancelled' => 'ملغية',
        ];

        return $labels[$this->status] ?? $this->status;
    }

    /**
     * Get the priority label.
     */
    public function getPriorityLabelAttribute()
    {
        $labels = [
            'low' => 'منخفضة',
            'medium' => 'متوسطة',
            'high' => 'عالية',
            'urgent' => 'عاجلة',
        ];

        return $labels[$this->priority] ?? $this->priority;
    }

    /**
     * Get the status color.
     */
    public function getStatusColorAttribute()
    {
        $colors = [
            'pending' => 'secondary',
            'in_progress' => 'primary',
            'completed' => 'success',
            'cancelled' => 'danger',
        ];

        return $colors[$this->status] ?? 'secondary';
    }

    /**
     * Get the priority color.
     */
    public function getPriorityColorAttribute()
    {
        $colors = [
            'low' => 'success',
            'medium' => 'info',
            'high' => 'warning',
            'urgent' => 'danger',
        ];

        return $colors[$this->priority] ?? 'secondary';
    }

    /**
     * Check if the task is overdue.
     */
    public function getIsOverdueAttribute()
    {
        if (in_array($this->status, ['completed', 'cancelled'])) {
            return false;
        }

        return $this->due_date && $this->due_date->isPast();
    }

    /**
     * Get the days remaining until the due date.
     */
    public function getDaysRemainingAttribute()
    {
        if (!$this->due_date) {
            return null;
        }

        $today = now()->startOfDay();

        if ($this->due_date->lt($today)) {
            return -$today->diffInDays($this->due_date);
        }

        return $today->diffInDays($this->due_date);
    }

    /**
     * Get the days since completion.
     */
    public function getDaysSinceCompletionAttribute()
    {
        if (!$this->completed_at) {
            return null;
        }

        return $this->completed_at->diffInDays(now());
    }

    /**
     * Get the formatted due date.
     */
    public function getFormattedDueDateAttribute()
    {
        return $this->due_date ? $this->due_date->format('Y-m-d') : null;
    }

    /**
     * Get the formatted completed date.
     */
    public function getFormattedCompletedAtAttribute()
    {
        return $this->completed_at ? $this->completed_at->format('Y-m-d H:i') : null;
    }

    /**
     * Get the employees assigned to the task.
     */
    public function employees()
    {
        return $this->belongsToMany(Employee::class, 'employee_task_assignees', 'task_id', 'employee_id');
    }
}
