<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;

class ElevatorPart extends Model
{
    use HasFactory;

    /**
     * الخصائص التي يمكن تعيينها بشكل جماعي.
     *
     * @var array
     */
    protected $fillable = [
        'company_id',
        'name',
        'part_number',
        'manufacturer',
        'model',
        'description',
        'specifications',
        'cost_price',
        'selling_price',
        'min_stock_level',
        'current_stock',
        'location',
        'unit',
        'is_active',
    ];

    /**
     * الخصائص التي يجب تحويلها.
     *
     * @var array
     */
    protected $casts = [
        'cost_price' => 'decimal:2',
        'selling_price' => 'decimal:2',
        'min_stock_level' => 'integer',
        'current_stock' => 'integer',
        'is_active' => 'boolean',
    ];

    /**
     * الحصول على الشركة المرتبطة بقطعة الغيار.
     */
    public function company(): BelongsTo
    {
        return $this->belongsTo(Company::class);
    }

    /**
     * التحقق مما إذا كان المخزون منخفض.
     *
     * @return bool
     */
    public function isLowStock(): bool
    {
        return $this->current_stock <= $this->min_stock_level;
    }

    /**
     * الحصول على حالة المخزون.
     *
     * @return string
     */
    public function getStockStatus(): string
    {
        if ($this->current_stock <= 0) {
            return 'نفذ';
        } elseif ($this->isLowStock()) {
            return 'منخفض';
        } else {
            return 'متوفر';
        }
    }

    /**
     * Get the maintenance operations that use this part.
     */
    public function maintenances(): BelongsToMany
    {
        return $this->belongsToMany(ElevatorMaintenance::class, 'maintenance_parts', 'part_id', 'maintenance_id')
            ->withPivot('quantity')
            ->withTimestamps();
    }
}
