<?php

/**
 * Created by Reliese Model.
 */

namespace App\Models;

use Carbon\Carbon;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use App\Models\ElevatorMaintenance;
use App\Models\InstallationProject;
use App\Models\ProjectElevator;

/**
 * Class Elevator
 * 
 * @property int $id
 * @property int $company_id
 * @property int $customer_id
 * @property int $location_id
 * @property int|null $user_id
 * @property int|null $branch_id
 * @property string $serial_number
 * @property string $model
 * @property string $manufacturer
 * @property Carbon $installation_date
 * @property Carbon|null $warranty_end_date
 * @property Carbon|null $free_maintenance_end_date
 * @property Carbon|null $control_warranty_end_date
 * @property Carbon|null $machine_warranty_end_date
 * @property Carbon|null $delivery_date
 * @property int $capacity
 * @property int $floors_count
 * @property float $speed
 * @property string $type
 * @property string $status
 * @property string|null $qr_code
 * @property string|null $specifications
 * @property string|null $notes
 * @property Carbon|null $created_at
 * @property Carbon|null $updated_at
 * @property string|null $deleted_at
 * 
 * @property Branch|null $branch
 * @property Company $company
 * @property Customer $customer
 * @property Location $location
 * @property User|null $user
 * @property Collection|ContractElevator[] $contract_elevators
 * @property Collection|MaintenanceLog[] $maintenance_logs
 * @property Collection|MaintenanceSchedule[] $maintenance_schedules
 * @property Collection|ProjectElevator[] $project_elevators
 *
 * @package App\Models
 */
class Elevator extends Model
{
	use SoftDeletes;
	protected $table = 'elevators';

	protected $casts = [
		'company_id' => 'int',
		'customer_id' => 'int',
		'location_id' => 'int',
		'user_id' => 'int',
		'branch_id' => 'int',
		'installation_date' => 'datetime',
		'warranty_end_date' => 'datetime',
		'free_maintenance_end_date' => 'datetime',
		'control_warranty_end_date' => 'datetime',
		'machine_warranty_end_date' => 'datetime',
		'delivery_date' => 'datetime',
		'capacity' => 'int',
		'floors_count' => 'int',
		'speed' => 'float',
		'price' => 'float'
	];

	protected $fillable = [
		'company_id',
		'customer_id',
		'location_id',
		'user_id',
		'branch_id',
		'serial_number',
		'model',
		'manufacturer',
		'installation_date',
		'warranty_end_date',
		'free_maintenance_end_date',
		'control_warranty_end_date',
		'machine_warranty_end_date',
		'delivery_date',
		'capacity',
		'floors_count',
		'speed',
		'price',
		'type',
		'status',
		'qr_code',
		'specifications',
		'notes'
	];

	public function branch()
	{
		return $this->belongsTo(Branch::class);
	}

	public function company()
	{
		return $this->belongsTo(Company::class);
	}

	public function customer()
	{
		return $this->belongsTo(Customer::class);
	}

	public function location()
	{
		return $this->belongsTo(Location::class);
	}

	public function user()
	{
		return $this->belongsTo(User::class);
	}

	public function contract_elevators()
	{
		return $this->hasMany(ContractElevator::class);
	}

	public function maintenance_logs()
	{
		return $this->hasMany(MaintenanceLog::class);
	}

	public function maintenance_schedules()
	{
		return $this->hasMany(MaintenanceSchedule::class);
	}

	public function project_elevators()
	{
		return $this->hasMany(ProjectElevator::class);
	}

	// Alias for location to maintain compatibility with code expecting 'building'
	public function building()
	{
		return $this->belongsTo(Location::class, 'location_id');
	}

	public function maintenanceContracts()
	{
		return $this->belongsToMany(MaintenanceContract::class, 'contract_elevator', 'elevator_id', 'maintenance_contract_id');
	}

	/**
	 * Get the maintenance operations for this elevator.
	 */
	public function maintenances()
	{
		return $this->hasMany(ElevatorMaintenance::class, 'elevator_id');
	}

	/**
	 * Get the installation project associated with this elevator.
	 */
	public function installationProject()
	{
		return $this->hasOneThrough(
			InstallationProject::class,
			ProjectElevator::class,
			'elevator_id', // Foreign key on ProjectElevator table
			'id', // Foreign key on InstallationProject table
			'id', // Local key on Elevator table
			'installation_project_id' // Local key on ProjectElevator table
		);
	}

	/**
	 * Check if the elevator is currently under warranty
	 * 
	 * @return bool
	 */
	public function isUnderWarranty()
	{
		if (!$this->warranty_end_date) {
			return false;
		}

		return $this->warranty_end_date->greaterThanOrEqualTo(now());
	}
}
