<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Log;

class CustomerStatementFixed extends Model
{
    use HasFactory;

    protected $table = 'customer_statements';

    protected $fillable = [
        'company_id',
        'customer_id',
        'branch_id',
        'transaction_date',
        'reference_number',
        'reference_type',
        'reference_id',
        'description',
        'debit',
        'credit',
        'balance',
    ];

    public static function createForJournalEntry(JournalEntry $journalEntry)
    {
        try {
            // أولاً، نحذف أي سجلات موجودة لهذا القيد المحاسبي
            self::where('reference_type', 'App\Models\JournalEntry')
                ->where('reference_id', $journalEntry->id)
                ->delete();

            // Find customer account items in this journal entry
            $customerAccountItems = JournalEntryItem::where('journal_entry_id', $journalEntry->id)
                ->whereHas('account', function ($query) {
                    $query->where('account_type', 'asset')
                        ->where(function ($q) {
                            $q->where('sub_type', 'accounts_receivable')
                                ->orWhere('sub_type', 'customer_receivable');
                        });
                })
                ->get();

            // If no customer account items found, we can't create a statement
            if ($customerAccountItems->isEmpty()) {
                return null;
            }

            $statementsCreated = [];

            // For each customer account item, create a statement
            foreach ($customerAccountItems as $item) {
                // Get the customer account
                $customerAccount = ChartOfAccount::find($item->account_id);
                if (!$customerAccount) continue;

                // Extract customer ID from account code (AR-{customer_id})
                $accountCode = $customerAccount->account_code;
                $customerId = null;

                if (strpos($accountCode, 'AR-') === 0) {
                    $customerId = intval(substr($accountCode, 3));
                }

                // If we couldn't extract customer ID, try to find by name
                if (!$customerId) {
                    // Extract customer name from account name (ذمم العميل: {customer_name})
                    $customerName = str_replace('ذمم العميل: ', '', $customerAccount->name);
                    $customerName = str_replace('Customer Receivable: ', '', $customerName);

                    // تحقق من وجود "عميل رقم" في اسم الحساب
                    if (preg_match('/عميل رقم(\d+)/', $customerName, $matches)) {
                        $customerId = intval($matches[1]);
                        Log::info('استخراج معرف العميل من اسم الحساب: ' . $customerName . ' -> ' . $customerId);
                    } else {
                        $customer = Customer::where('company_id', $journalEntry->company_id)
                            ->where(function ($query) use ($customerName) {
                                $query->where('name', 'like', '%' . $customerName . '%')
                                    ->orWhere('name_en', 'like', '%' . $customerName . '%');
                            })
                            ->first();

                        if ($customer) {
                            $customerId = $customer->id;
                            Log::info('العثور على العميل بالاسم: ' . $customerName . ' -> ' . $customerId);
                        } else {
                            // Try to find customer by searching all customers and comparing with account name
                            $allCustomers = Customer::where('company_id', $journalEntry->company_id)->get();
                            foreach ($allCustomers as $potentialCustomer) {
                                if (
                                    stripos($customerAccount->name, $potentialCustomer->name) !== false ||
                                    (isset($potentialCustomer->name_en) && stripos($customerAccount->name, $potentialCustomer->name_en) !== false)
                                ) {
                                    $customerId = $potentialCustomer->id;
                                    Log::info('العثور على العميل بمقارنة الاسم: ' . $customerAccount->name . ' -> ' . $customerId);
                                    break;
                                }
                            }

                            if (!$customerId) {
                                Log::warning('لم يتم العثور على العميل للحساب: ' . $customerAccount->name . ' (' . $accountCode . ')');
                                continue; // Skip if we can't find the customer
                            }
                        }
                    }
                } else {
                    // Verify customer exists
                    $customer = Customer::find($customerId);
                    if (!$customer) {
                        Log::warning('العميل غير موجود: ' . $customerId);
                        continue;
                    }
                    Log::info('استخراج معرف العميل من رمز الحساب: ' . $accountCode . ' -> ' . $customerId);
                }

                // Get the last balance for this customer
                $lastStatement = self::where('customer_id', $customerId)
                    ->orderBy('id', 'desc')
                    ->first();

                $currentBalance = $lastStatement ? $lastStatement->balance : 0;

                // Calculate the new balance
                // If debit > 0, it increases the customer balance (customer owes more)
                // If credit > 0, it decreases the customer balance (customer paid)
                $debitAmount = $item->debit;
                $creditAmount = $item->credit;
                $netAmount = $creditAmount - $debitAmount;

                // If net amount is credit (positive), it reduces the customer balance
                // If net amount is debit (negative), it increases the customer balance
                $newBalance = $currentBalance - $netAmount;

                // Get reference information based on journal entry reference
                $referenceType = $journalEntry->reference_type;
                $referenceId = $journalEntry->reference_id;
                $description = $journalEntry->description;

                // If reference is an invoice, get more details
                if ($referenceType == 'App\Models\Invoice' && $referenceId) {
                    $invoice = Invoice::find($referenceId);
                    if ($invoice) {
                        $description = 'فاتورة رقم ' . $invoice->invoice_number . ' (قيد محاسبي: ' . $journalEntry->entry_number . ')';
                    }
                }
                // If reference is a payment, get more details
                else if ($referenceType == 'App\Models\Payment' && $referenceId) {
                    $payment = Payment::find($referenceId);
                    if ($payment && $payment->invoice) {
                        $description = 'دفعة للفاتورة رقم ' . $payment->invoice->invoice_number . ' (قيد محاسبي: ' . $journalEntry->entry_number . ')';
                    }
                }

                // Create the statement record
                $statement = self::create([
                    'company_id' => $journalEntry->company_id,
                    'customer_id' => $customerId,
                    'branch_id' => null, // We don't have branch info from journal entry
                    'transaction_date' => $journalEntry->entry_date,
                    'reference_number' => $journalEntry->entry_number,
                    'reference_type' => 'App\Models\JournalEntry',
                    'reference_id' => $journalEntry->id,
                    'description' => $description,
                    'debit' => $debitAmount,
                    'credit' => $creditAmount,
                    'balance' => $newBalance
                ]);

                $statementsCreated[] = $statement;
            }

            return count($statementsCreated) > 0 ? $statementsCreated[0] : null;
        } catch (\Exception $e) {
            Log::error('Error creating customer statement from journal entry: ' . $e->getMessage());
            return null;
        }
    }
}
