<?php

/**
 * Created by Reliese Model.
 */

namespace App\Models;

use Carbon\Carbon;
use Illuminate\Database\Eloquent\Model;
use App\Models\JournalEntry;
use App\Models\JournalEntryItem;
use App\Models\ChartOfAccount;
use App\Models\Customer;
use App\Models\Invoice;
use App\Models\Payment;

/**
 * Class CustomerStatement
 * 
 * @property int $id
 * @property int $company_id
 * @property int $customer_id
 * @property int|null $branch_id
 * @property Carbon $transaction_date
 * @property string $reference_number
 * @property string $reference_type
 * @property int $reference_id
 * @property string $description
 * @property float $debit
 * @property float $credit
 * @property float $balance
 * @property Carbon|null $created_at
 * @property Carbon|null $updated_at
 * 
 * @property Branch|null $branch
 * @property Company $company
 * @property Customer $customer
 *
 * @package App\Models
 */
class CustomerStatement extends Model
{
	protected $table = 'customer_statements';

	protected $casts = [
		'company_id' => 'int',
		'customer_id' => 'int',
		'branch_id' => 'int',
		'transaction_date' => 'datetime',
		'reference_id' => 'int',
		'debit' => 'float',
		'credit' => 'float',
		'balance' => 'float'
	];

	protected $fillable = [
		'company_id',
		'customer_id',
		'branch_id',
		'transaction_date',
		'reference_number',
		'reference_type',
		'reference_id',
		'description',
		'debit',
		'credit',
		'balance'
	];

	/**
	 * Create a customer statement record for a payment
	 * 
	 * @param Payment $payment
	 * @return CustomerStatement
	 */
	public static function createForPayment(Payment $payment)
	{
		// Get the invoice associated with the payment
		$invoice = $payment->invoice;

		// Get the customer from the invoice
		$customer = $invoice->customer;

		// Get the customer's account from chart of accounts
		$customerAccount = \App\Models\ChartOfAccount::where('company_id', $payment->company_id)
			->where('account_type', 'asset')
			->where('sub_type', 'accounts_receivable')
			->where('name', 'like', '%' . $customer->name . '%')
			->first();

		// If no customer account found, create one
		if (!$customerAccount) {
			// Find the main accounts receivable account
			$mainAccountReceivable = \App\Models\ChartOfAccount::where('company_id', $payment->company_id)
				->where('account_type', 'asset')
				->where('sub_type', 'accounts_receivable')
				->where('is_parent', true)
				->first();

			// If main account not found, we can't proceed
			if (!$mainAccountReceivable) {
				// Fallback to original implementation
				return self::createStatementFromPayment($payment);
			}

			// Create a new account for this customer
			$customerAccount = new \App\Models\ChartOfAccount();
			$customerAccount->company_id = $payment->company_id;
			$customerAccount->account_code = 'AR-' . $customer->id;
			$customerAccount->name = 'ذمم العميل: ' . $customer->name;
			$customerAccount->name_en = 'Customer Receivable: ' . $customer->name;
			$customerAccount->account_type = 'asset';
			$customerAccount->sub_type = 'accounts_receivable';
			$customerAccount->parent_id = $mainAccountReceivable->id;
			$customerAccount->description = 'حساب ذمم العميل: ' . $customer->name;
			$customerAccount->is_active = true;
			$customerAccount->is_system = false;
			$customerAccount->opening_balance = 0;
			$customerAccount->current_balance = 0;
			$customerAccount->currency = 'SAR';
			$customerAccount->balance_type = 'debit';
			$customerAccount->is_cash_account = false;
			$customerAccount->is_bank_account = false;
			$customerAccount->level = $mainAccountReceivable->level + 1;
			$customerAccount->is_parent = false;
			$customerAccount->save();
		}

		// Find journal entries related to this payment
		$journalEntries = \App\Models\JournalEntry::where('reference_type', 'App\\Models\\Payment')
			->where('reference_id', $payment->id)
			->get();

		// If no journal entries found for this payment, create one
		if ($journalEntries->isEmpty()) {
			try {
				// Get the cash/bank account based on payment method
				$cashAccount = \App\Models\ChartOfAccount::where('company_id', $payment->company_id)
					->where(function ($query) {
						$query->where('is_cash_account', true)
							->orWhere('is_bank_account', true);
					})
					->first();

				if (!$cashAccount) {
					// Fallback to original implementation if no cash account found
					return self::createStatementFromPayment($payment);
				}

				// Get fiscal year and accounting period
				$fiscalYear = \App\Models\FiscalYear::where('company_id', $payment->company_id)
					->where('is_active', true)
					->where('is_closed', false)
					->first();

				if (!$fiscalYear) {
					// Fallback to original implementation if no fiscal year found
					return self::createStatementFromPayment($payment);
				}

				$accountingPeriod = \App\Models\AccountingPeriod::where('fiscal_year_id', $fiscalYear->id)
					->where('is_closed', false)
					->where('start_date', '<=', $payment->payment_date)
					->where('end_date', '>=', $payment->payment_date)
					->first();

				if (!$accountingPeriod) {
					// Fallback to original implementation if no accounting period found
					return self::createStatementFromPayment($payment);
				}

				// Create a new journal entry for this payment
				$journalEntry = new \App\Models\JournalEntry();
				$journalEntry->company_id = $payment->company_id;
				$journalEntry->fiscal_year_id = $fiscalYear->id;
				$journalEntry->accounting_period_id = $accountingPeriod->id;
				$journalEntry->entry_number = \App\Models\JournalEntry::generateEntryNumber($payment->company_id);
				$journalEntry->entry_date = $payment->payment_date;
				$journalEntry->reference_type = 'App\\Models\\Payment';
				$journalEntry->reference_id = $payment->id;
				$journalEntry->description = 'دفعة للفاتورة رقم ' . $invoice->invoice_number;
				$journalEntry->is_posted = false;
				$journalEntry->is_approved = false;
				$journalEntry->created_by = $payment->created_by;
				$journalEntry->save();

				// Create debit entry (cash/bank)
				$debitItem = new \App\Models\JournalEntryItem();
				$debitItem->journal_entry_id = $journalEntry->id;
				$debitItem->account_id = $cashAccount->id;
				$debitItem->description = 'دفعة للفاتورة رقم ' . $invoice->invoice_number;
				$debitItem->debit = $payment->amount;
				$debitItem->credit = 0;
				$debitItem->save();

				// Create credit entry (customer account)
				$creditItem = new \App\Models\JournalEntryItem();
				$creditItem->journal_entry_id = $journalEntry->id;
				$creditItem->account_id = $customerAccount->id;
				$creditItem->description = 'دفعة للفاتورة رقم ' . $invoice->invoice_number . ' - ' . $customer->name;
				$creditItem->debit = 0;
				$creditItem->credit = $payment->amount;
				$creditItem->save();

				// Post the journal entry
				$journalEntry->post();

				// Add the new journal entry to our collection
				$journalEntries->push($journalEntry);
			} catch (\Exception $e) {
				// If there's an error creating the journal entry, log it and continue with original implementation
				\Illuminate\Support\Facades\Log::error('Error creating journal entry for payment: ' . $e->getMessage());
				return self::createStatementFromPayment($payment);
			}
		}

		// Find journal entry items that affect this customer's account
		$journalEntryItems = \App\Models\JournalEntryItem::where('account_id', $customerAccount->id)
			->whereIn('journal_entry_id', $journalEntries->pluck('id'))
			->get();

		// If no items found, use original implementation
		if ($journalEntryItems->isEmpty()) {
			return self::createStatementFromPayment($payment);
		}

		// Get the last balance for this customer
		$lastStatement = self::where('customer_id', $customer->id)
			->orderBy('id', 'desc')
			->first();

		$currentBalance = $lastStatement ? $lastStatement->balance : 0;

		// Calculate the new balance based on journal entry items
		$creditAmount = $journalEntryItems->sum('credit');
		$debitAmount = $journalEntryItems->sum('debit');
		$netAmount = $creditAmount - $debitAmount;

		// If net amount is credit (positive), it reduces the customer balance
		$newBalance = $currentBalance - $netAmount;

		// Create the new statement record
		$statement = self::create([
			'company_id' => $payment->company_id,
			'customer_id' => $customer->id,
			'branch_id' => $invoice->branch_id,
			'transaction_date' => $payment->payment_date,
			'reference_number' => $journalEntries->first()->entry_number,
			'reference_type' => 'App\\Models\\JournalEntry',
			'reference_id' => $journalEntries->first()->id,
			'description' => 'دفعة للفاتورة رقم ' . $invoice->invoice_number . ' (قيد محاسبي: ' . $journalEntries->pluck('entry_number')->implode(', ') . ')',
			'debit' => $debitAmount,
			'credit' => $creditAmount,
			'balance' => $newBalance
		]);

		return $statement;
	}

	/**
	 * Fallback method to create a statement directly from payment
	 * 
	 * @param Payment $payment
	 * @return CustomerStatement
	 */
	private static function createStatementFromPayment(Payment $payment)
	{
		// Get the invoice associated with the payment
		$invoice = $payment->invoice;

		// Get the customer from the invoice
		$customer = $invoice->customer;

		// Get the last balance for this customer
		$lastStatement = self::where('customer_id', $customer->id)
			->orderBy('id', 'desc')
			->first();

		$currentBalance = $lastStatement ? $lastStatement->balance : 0;

		// Calculate the new balance (subtract payment amount from current balance)
		$newBalance = $currentBalance - $payment->amount;

		// Create the new statement record
		return self::create([
			'company_id' => $payment->company_id,
			'customer_id' => $customer->id,
			'branch_id' => $invoice->branch_id,
			'transaction_date' => $payment->payment_date,
			'reference_number' => $payment->reference_number,
			'reference_type' => 'App\\Models\\Payment',
			'reference_id' => $payment->id,
			'description' => 'دفعة للفاتورة رقم ' . $invoice->invoice_number,
			'debit' => 0,
			'credit' => $payment->amount,
			'balance' => $newBalance
		]);
	}

	public function branch()
	{
		return $this->belongsTo(Branch::class);
	}

	public function company()
	{
		return $this->belongsTo(Company::class);
	}

	public function customer()
	{
		return $this->belongsTo(Customer::class);
	}

	/**
	 * Create a customer statement record for a journal entry
	 * 
	 * @param JournalEntry $journalEntry
	 * @return CustomerStatement|null
	 */
	public static function createForJournalEntry(JournalEntry $journalEntry)
	{
		try {
			// أولاً، نحذف أي سجلات موجودة لهذا القيد المحاسبي
			self::where('reference_type', 'App\Models\JournalEntry')
				->where('reference_id', $journalEntry->id)
				->delete();

			// Find customer account items in this journal entry
			$customerAccountItems = JournalEntryItem::where('journal_entry_id', $journalEntry->id)
				->whereHas('account', function ($query) {
					$query->where('account_type', 'asset')
						->where(function ($q) {
							$q->where('sub_type', 'accounts_receivable')
								->orWhere('sub_type', 'customer_receivable');
						});
				})
				->get();

			// If no customer account items found, we can't create a statement
			if ($customerAccountItems->isEmpty()) {
				return null;
			}

			$statementsCreated = [];

			// For each customer account item, create a statement
			foreach ($customerAccountItems as $item) {
				// Get the customer account
				$customerAccount = ChartOfAccount::find($item->account_id);
				if (!$customerAccount) continue;

				// Extract customer ID from account code (AR-{customer_id})
				$accountCode = $customerAccount->account_code;
				$customerId = null;

				if (strpos($accountCode, 'AR-') === 0) {
					$customerId = intval(substr($accountCode, 3));
				}

				// If we couldn't extract customer ID, try to find by name
				if (!$customerId) {
					// Extract customer name from account name (ذمم العميل: {customer_name})
					$customerName = str_replace('ذمم العميل: ', '', $customerAccount->name);
					$customerName = str_replace('Customer Receivable: ', '', $customerName);

					$customer = Customer::where('company_id', $journalEntry->company_id)
						->where(function ($query) use ($customerName) {
							$query->where('name', 'like', '%' . $customerName . '%')
								->orWhere('name_en', 'like', '%' . $customerName . '%');
						})
						->first();

					if ($customer) {
						$customerId = $customer->id;
					} else {
						// Try to find customer by searching all customers and comparing with account name
						$allCustomers = Customer::where('company_id', $journalEntry->company_id)->get();
						foreach ($allCustomers as $potentialCustomer) {
							if (
								stripos($customerAccount->name, $potentialCustomer->name) !== false ||
								(isset($potentialCustomer->name_en) && stripos($customerAccount->name, $potentialCustomer->name_en) !== false)
							) {
								$customerId = $potentialCustomer->id;
								break;
							}
						}

						if (!$customerId) {
							continue; // Skip if we can't find the customer
						}
					}
				} else {
					// Verify customer exists
					$customer = Customer::find($customerId);
					if (!$customer) continue;
				}

				// Get the last balance for this customer
				$lastStatement = self::where('customer_id', $customerId)
					->orderBy('id', 'desc')
					->first();

				$currentBalance = $lastStatement ? $lastStatement->balance : 0;

				// Calculate the new balance
				// If debit > 0, it increases the customer balance (customer owes more)
				// If credit > 0, it decreases the customer balance (customer paid)
				$debitAmount = $item->debit;
				$creditAmount = $item->credit;
				$netAmount = $creditAmount - $debitAmount;

				// If net amount is credit (positive), it reduces the customer balance
				// If net amount is debit (negative), it increases the customer balance
				$newBalance = $currentBalance - $netAmount;

				// Get reference information based on journal entry reference
				$referenceType = $journalEntry->reference_type;
				$referenceId = $journalEntry->reference_id;
				$description = $journalEntry->description;

				// If reference is an invoice, get more details
				if ($referenceType == 'App\Models\Invoice' && $referenceId) {
					$invoice = Invoice::find($referenceId);
					if ($invoice) {
						$description = 'فاتورة رقم ' . $invoice->invoice_number . ' (قيد محاسبي: ' . $journalEntry->entry_number . ')';
					}
				}
				// If reference is a payment, get more details
				else if ($referenceType == 'App\Models\Payment' && $referenceId) {
					$payment = Payment::find($referenceId);
					if ($payment && $payment->invoice) {
						$description = 'دفعة للفاتورة رقم ' . $payment->invoice->invoice_number . ' (قيد محاسبي: ' . $journalEntry->entry_number . ')';
					}
				}

				// Create the statement record
				$statement = self::create([
					'company_id' => $journalEntry->company_id,
					'customer_id' => $customerId,
					'branch_id' => null, // We don't have branch info from journal entry
					'transaction_date' => $journalEntry->entry_date,
					'reference_number' => $journalEntry->entry_number,
					'reference_type' => 'App\Models\JournalEntry',
					'reference_id' => $journalEntry->id,
					'description' => $description,
					'debit' => $debitAmount,
					'credit' => $creditAmount,
					'balance' => $newBalance
				]);

				$statementsCreated[] = $statement;
			}

			return count($statementsCreated) > 0 ? $statementsCreated[0] : null;
		} catch (\Exception $e) {
			\Illuminate\Support\Facades\Log::error('Error creating customer statement from journal entry: ' . $e->getMessage());
			return null;
		}
	}
}
