<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class Cheque extends Model
{
    use HasFactory, SoftDeletes;

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'company_id',
        'type',
        'cheque_number',
        'cheque_date',
        'amount',
        'currency',
        'bank_id',
        'bank_account_id',
        'drawer_name',
        'beneficiary_name',
        'status',
        'due_date',
        'deposit_date',
        'clearing_date',
        'reference_type',
        'reference_id',
        'bank_transaction_id',
        'notes',
        'attachment',
        'created_by',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array
     */
    protected $casts = [
        'cheque_date' => 'date',
        'amount' => 'decimal:2',
        'due_date' => 'date',
        'deposit_date' => 'date',
        'clearing_date' => 'date',
    ];

    /**
     * Get the company that owns the cheque.
     */
    public function company()
    {
        return $this->belongsTo(Company::class);
    }

    /**
     * Get the bank that owns the cheque.
     */
    public function bank()
    {
        return $this->belongsTo(Bank::class);
    }

    /**
     * Get the bank account that owns the cheque.
     */
    public function bankAccount()
    {
        return $this->belongsTo(BankAccount::class);
    }

    /**
     * Get the bank transaction associated with the cheque.
     */
    public function bankTransaction()
    {
        return $this->belongsTo(BankTransaction::class);
    }

    /**
     * Get the user who created the cheque.
     */
    public function createdBy()
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    /**
     * Get the bank reconciliation items for the cheque.
     */
    public function bankReconciliationItems()
    {
        return $this->hasMany(BankReconciliationItem::class);
    }

    /**
     * Get the related model based on reference_type and reference_id.
     */
    public function reference()
    {
        if (!$this->reference_type || !$this->reference_id) {
            return null;
        }

        $modelClass = '\\App\\Models\\' . $this->reference_type;
        if (!class_exists($modelClass)) {
            return null;
        }

        return $modelClass::find($this->reference_id);
    }
}
