<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class ChartOfAccount extends Model
{
    use HasFactory, SoftDeletes;

    /**
     * اسم الجدول المرتبط بالنموذج
     *
     * @var string
     */
    protected $table = 'chart_of_accounts';

    /**
     * الحقول القابلة للتعبئة الجماعية
     *
     * @var array
     */
    protected $fillable = [
        'company_id',
        'account_code',
        'name',
        'name_en',
        'account_type',
        'sub_type',
        'parent_id',
        'description',
        'is_active',
        'is_system',
        'opening_balance',
        'current_balance',
        'currency',
        'balance_type',
        'is_cash_account',
        'is_bank_account',
        'level',
        'is_parent'
    ];

    /**
     * الحقول التي يجب تحويلها إلى أنواع محددة
     *
     * @var array
     */
    protected $casts = [
        'opening_balance' => 'decimal:2',
        'current_balance' => 'decimal:2',
        'is_active' => 'boolean',
        'is_cash_account' => 'boolean',
        'is_bank_account' => 'boolean',
        'is_parent' => 'boolean'
    ];

    /**
     * علاقة مع الشركة
     */
    public function company()
    {
        return $this->belongsTo(Company::class);
    }

    /**
     * علاقة مع الحساب الأب
     */
    public function parent()
    {
        return $this->belongsTo(ChartOfAccount::class, 'parent_id');
    }

    /**
     * علاقة مع الحسابات الفرعية
     */
    public function children()
    {
        return $this->hasMany(ChartOfAccount::class, 'parent_id');
    }

    /**
     * علاقة مع بنود القيود المحاسبية
     */
    public function journalItems()
    {
        return $this->hasMany(JournalEntryItem::class, 'account_id');
    }

    /**
     * علاقة مع الخزائن
     */
    public function cashRegisters()
    {
        return $this->belongsToMany(CashRegister::class, 'cash_register_accounts', 'account_id', 'cash_register_id');
    }

    /**
     * نطاق للحسابات النشطة
     */
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    /**
     * نطاق للحسابات الرئيسية (بدون أب)
     */
    public function scopeParents($query)
    {
        return $query->whereNull('parent_id');
    }

    /**
     * نطاق لحسابات الأصول
     */
    public function scopeAssets($query)
    {
        return $query->where('account_type', 'asset');
    }

    /**
     * نطاق لحسابات الخصوم
     */
    public function scopeLiabilities($query)
    {
        return $query->where('account_type', 'liability');
    }

    /**
     * نطاق لحسابات الإيرادات
     */
    public function scopeRevenues($query)
    {
        return $query->where('account_type', 'revenue');
    }

    /**
     * نطاق لحسابات المصروفات
     */
    public function scopeExpenses($query)
    {
        return $query->where('account_type', 'expense');
    }

    /**
     * نطاق لحسابات حقوق الملكية
     */
    public function scopeEquity($query)
    {
        return $query->where('account_type', 'equity');
    }

    /**
     * نطاق لحسابات النقدية
     */
    public function scopeCash($query)
    {
        return $query->where('is_cash_account', true);
    }

    /**
     * نطاق لحسابات البنوك
     */
    public function scopeBank($query)
    {
        return $query->where('is_bank_account', true);
    }

    /**
     * الحصول على الرصيد الحالي بتنسيق العملة
     */
    public function getFormattedBalanceAttribute()
    {
        return number_format($this->current_balance, 2) . ' ' . $this->currency;
    }

    /**
     * تحديث الرصيد الحالي للحساب
     */
    public function updateBalance()
    {
        $debits = $this->journalItems()->sum('debit');
        $credits = $this->journalItems()->sum('credit');

        // تحديث الرصيد بناءً على نوع الحساب
        if (in_array($this->account_type, ['asset', 'expense'])) {
            // الأصول والمصروفات تزيد بالمدين وتنقص بالدائن
            $this->current_balance = $this->opening_balance + $debits - $credits;
        } else {
            // الخصوم والإيرادات وحقوق الملكية تزيد بالدائن وتنقص بالمدين
            $this->current_balance = $this->opening_balance + $credits - $debits;
        }

        $this->save();

        // إذا كان هناك حساب أب، نقوم بتحديث رصيده أيضًا
        if ($this->parent_id) {
            $this->parent->updateBalance();
        }
    }

    /**
     * الحصول على حركة الحساب بين تاريخين محددين
     * 
     * @param \Carbon\Carbon|string $startDate تاريخ البداية
     * @param \Carbon\Carbon|string $endDate تاريخ النهاية
     * @return array مصفوفة تحتوي على إجمالي المدين والدائن
     */
    public function getMovementBetweenDates($startDate, $endDate)
    {
        // التأكد من تحويل التواريخ إلى كائنات Carbon إذا كانت نصوص
        if (is_string($startDate)) {
            $startDate = \Carbon\Carbon::parse($startDate);
        }

        if (is_string($endDate)) {
            $endDate = \Carbon\Carbon::parse($endDate);
        }

        // الحصول على بنود القيود المحاسبية المرتبطة بالحساب في الفترة المحددة
        $items = $this->journalItems()
            ->whereHas('journalEntry', function ($query) use ($startDate, $endDate) {
                $query->where('is_posted', true)
                    ->whereBetween('entry_date', [$startDate, $endDate]);
            })
            ->get();

        // حساب إجمالي المدين والدائن
        $debit = $items->sum('debit');
        $credit = $items->sum('credit');

        // إذا كان الحساب له حسابات فرعية، نضيف حركتها أيضًا
        $childrenAccounts = $this->children;
        foreach ($childrenAccounts as $childAccount) {
            $childMovement = $childAccount->getMovementBetweenDates($startDate, $endDate);
            $debit += $childMovement['debit'];
            $credit += $childMovement['credit'];
        }

        return [
            'debit' => $debit,
            'credit' => $credit
        ];
    }

    /**
     * الحصول على رصيد الحساب في تاريخ محدد
     * 
     * @param \Carbon\Carbon|string $date التاريخ
     * @return float رصيد الحساب
     */
    public function getBalanceAtDate($date)
    {
        // التأكد من تحويل التاريخ إلى كائن Carbon إذا كان نص
        if (is_string($date)) {
            $date = \Carbon\Carbon::parse($date);
        }

        // الحصول على بنود القيود المحاسبية المرتبطة بالحساب حتى التاريخ المحدد
        $items = $this->journalItems()
            ->whereHas('journalEntry', function ($query) use ($date) {
                $query->where('is_posted', true)
                    ->whereDate('entry_date', '<=', $date);
            })
            ->get();

        // حساب إجمالي المدين والدائن
        $debit = $items->sum('debit');
        $credit = $items->sum('credit');

        // حساب الرصيد بناءً على نوع الحساب
        $balance = 0;

        if (in_array($this->account_type, ['asset', 'expense'])) {
            // الأصول والمصروفات تزيد بالمدين وتنقص بالدائن
            $balance = $this->opening_balance + $debit - $credit;
        } else {
            // الخصوم والإيرادات وحقوق الملكية تزيد بالدائن وتنقص بالمدين
            $balance = $this->opening_balance + $credit - $debit;
        }

        // إذا كان الحساب له حسابات فرعية، نضيف أرصدتها أيضًا
        $childrenAccounts = $this->children;
        foreach ($childrenAccounts as $childAccount) {
            $balance += $childAccount->getBalanceAtDate($date);
        }

        return $balance;
    }
}
