<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class CashTransaction extends Model
{
    use HasFactory, SoftDeletes;

    /**
     * الحقول القابلة للتعبئة الجماعية
     *
     * @var array
     */
    protected $fillable = [
        'cash_register_id',
        'company_id',
        'user_id',
        'type',
        'amount',
        'receipt_date',
        'related_cash_register_id',
        'party_name',
        'party_type',
        'reference_type',
        'reference_id',
        'elevator_id',
        'notes',
        'attachment'
    ];

    /**
     * الحقول التي يجب تحويلها إلى أنواع محددة
     *
     * @var array
     */
    protected $casts = [
        'amount' => 'decimal:2',
        'receipt_date' => 'date',
    ];

    /**
     * علاقة مع الخزينة
     */
    public function cashRegister()
    {
        return $this->belongsTo(CashRegister::class);
    }

    /**
     * علاقة مع الخزينة المرتبطة (في حالة التحويل)
     */
    public function relatedCashRegister()
    {
        return $this->belongsTo(CashRegister::class, 'related_cash_register_id');
    }

    /**
     * علاقة مع الشركة
     */
    public function company()
    {
        return $this->belongsTo(Company::class);
    }

    /**
     * علاقة مع المستخدم
     */
    public function user()
    {
        return $this->belongsTo(User::class);
    }

    /**
     * علاقة مع المرجع (مورفيك)
     */
    public function reference()
    {
        return $this->morphTo();
    }

    /**
     * علاقة مع المصعد
     */
    public function elevator()
    {
        return $this->belongsTo(Elevator::class);
    }

    /**
     * نطاق للإيداعات
     */
    public function scopeDeposits($query)
    {
        return $query->where('type', 'deposit');
    }

    /**
     * نطاق للسحوبات
     */
    public function scopeWithdrawals($query)
    {
        return $query->where('type', 'withdrawal');
    }

    /**
     * نطاق للتحويلات
     */
    public function scopeTransfers($query)
    {
        return $query->where('type', 'transfer');
    }

    /**
     * الحصول على اسم نوع المعاملة بالعربية
     */
    public function getTypeNameAttribute()
    {
        $types = [
            'deposit' => 'إيداع',
            'withdrawal' => 'سحب',
            'transfer' => 'تحويل'
        ];

        return $types[$this->type] ?? $this->type;
    }
}
