<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class CashRegister extends Model
{
    use HasFactory, SoftDeletes;

    /**
     * الحقول القابلة للتعبئة الجماعية
     *
     * @var array
     */
    protected $fillable = [
        'name',
        'code',
        'description',
        'company_id',
        'branch_id',
        'responsible_user_id',
        'initial_balance',
        'current_balance',
        'currency',
        'is_active',
        'is_main'
    ];

    /**
     * الحقول التي يجب تحويلها إلى أنواع محددة
     *
     * @var array
     */
    protected $casts = [
        'initial_balance' => 'decimal:2',
        'current_balance' => 'decimal:2',
        'is_active' => 'boolean',
        'is_main' => 'boolean',
    ];

    /**
     * علاقة مع الشركة
     */
    public function company()
    {
        return $this->belongsTo(Company::class);
    }

    /**
     * علاقة مع الفرع
     */
    public function branch()
    {
        return $this->belongsTo(Branch::class);
    }

    /**
     * علاقة مع المستخدم المسؤول
     */
    public function responsibleUser()
    {
        return $this->belongsTo(User::class, 'responsible_user_id');
    }

    /**
     * علاقة مع معاملات الخزينة
     */
    public function transactions()
    {
        return $this->hasMany(CashTransaction::class);
    }

    /**
     * علاقة مع معاملات الخزينة المرتبطة (التحويلات)
     */
    public function relatedTransactions()
    {
        return $this->hasMany(CashTransaction::class, 'related_cash_register_id');
    }

    /**
     * الحصول على إجمالي الإيداعات
     */
    public function getTotalDepositsAttribute()
    {
        return $this->transactions()->where('type', 'deposit')->sum('amount');
    }

    /**
     * الحصول على إجمالي السحوبات
     */
    public function getTotalWithdrawalsAttribute()
    {
        return $this->transactions()->where('type', 'withdrawal')->sum('amount');
    }

    /**
     * الحصول على إجمالي التحويلات الصادرة
     */
    public function getTotalOutgoingTransfersAttribute()
    {
        return $this->transactions()->where('type', 'transfer')->sum('amount');
    }

    /**
     * الحصول على إجمالي التحويلات الواردة
     */
    public function getTotalIncomingTransfersAttribute()
    {
        return $this->relatedTransactions()->where('type', 'transfer')->sum('amount');
    }

    /**
     * علاقة مع حسابات الخزينة
     */
    public function cashRegisterAccounts()
    {
        return $this->hasMany(CashRegisterAccount::class);
    }

    /**
     * الحصول على الحساب المرتبط بالخزينة
     */
    public function getAccountAttribute()
    {
        return $this->cashRegisterAccounts()->first()?->account;
    }
}
