<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class BankTransaction extends Model
{
    use HasFactory, SoftDeletes;

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'company_id',
        'bank_account_id',
        'transaction_number',
        'transaction_type',
        'transaction_date',
        'amount',
        'related_bank_account_id',
        'related_cash_register_id',
        'related_transaction_id',
        'reference_number',
        'reference_type',
        'reference_id',
        'description',
        'attachment',
        'created_by',
        'approved_by',
        'approved_at',
        'is_reconciled',
        'journal_entry_id',
        'status',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array
     */
    protected $casts = [
        'transaction_date' => 'date',
        'amount' => 'decimal:2',
        'approved_at' => 'datetime',
        'is_reconciled' => 'boolean',
        'status' => 'string',
    ];

    /**
     * Get the company that owns the bank transaction.
     */
    public function company()
    {
        return $this->belongsTo(Company::class);
    }

    /**
     * Get the bank account that owns the bank transaction.
     */
    public function bankAccount()
    {
        return $this->belongsTo(BankAccount::class);
    }

    /**
     * Get the related bank account for the bank transaction.
     */
    public function relatedBankAccount()
    {
        return $this->belongsTo(BankAccount::class, 'related_bank_account_id');
    }

    /**
     * Get the related transaction for bank transfers.
     */
    public function relatedTransaction()
    {
        return $this->belongsTo(BankTransaction::class, 'related_transaction_id');
    }

    /**
     * Get the related cash register for the bank transaction.
     */
    public function relatedCashRegister()
    {
        return $this->belongsTo(CashRegister::class, 'related_cash_register_id');
    }

    /**
     * Get the user who created the bank transaction.
     */
    public function createdBy()
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    /**
     * Get the user who approved the bank transaction.
     */
    public function approvedBy()
    {
        return $this->belongsTo(User::class, 'approved_by');
    }

    /**
     * Get the cheque associated with the bank transaction.
     */
    public function cheque()
    {
        return $this->hasOne(Cheque::class);
    }

    /**
     * Get the journal entries for the bank transaction.
     */
    public function journalEntries()
    {
        return $this->belongsToMany(JournalEntry::class, 'bank_transaction_entries');
    }

    /**
     * Get the direct journal entry associated with the bank transaction.
     */
    public function journalEntry()
    {
        return $this->belongsTo(JournalEntry::class);
    }

    /**
     * Get the bank reconciliation items for the bank transaction.
     */
    public function bankReconciliationItems()
    {
        return $this->hasMany(BankReconciliationItem::class);
    }

    /**
     * Get the related model based on reference_type and reference_id.
     */
    public function reference()
    {
        if (!$this->reference_type || !$this->reference_id) {
            return null;
        }

        $modelClass = '\\App\\Models\\' . $this->reference_type;
        if (!class_exists($modelClass)) {
            return null;
        }

        return $modelClass::find($this->reference_id);
    }
}
