<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class BankAccount extends Model
{
    use HasFactory, SoftDeletes;

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'company_id',
        'bank_id',
        'chart_account_id',
        'account_number',
        'iban',
        'account_name',
        'account_type',
        'currency',
        'opening_balance',
        'current_balance',
        'opening_date',
        'is_active',
        'notes',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array
     */
    protected $casts = [
        'opening_balance' => 'decimal:2',
        'current_balance' => 'decimal:2',
        'opening_date' => 'date',
        'is_active' => 'boolean',
    ];

    /**
     * Get the company that owns the bank account.
     */
    public function company()
    {
        return $this->belongsTo(Company::class);
    }

    /**
     * Get the bank that owns the bank account.
     */
    public function bank()
    {
        return $this->belongsTo(Bank::class);
    }

    /**
     * Get the chart of account associated with the bank account.
     */
    public function chartOfAccount()
    {
        return $this->belongsTo(ChartOfAccount::class, 'chart_account_id');
    }

    /**
     * Get the bank transactions for the bank account.
     */
    public function bankTransactions()
    {
        return $this->hasMany(BankTransaction::class);
    }

    /**
     * Get the related bank transactions for the bank account.
     */
    public function relatedBankTransactions()
    {
        return $this->hasMany(BankTransaction::class, 'related_bank_account_id');
    }

    /**
     * Get the cheques for the bank account.
     */
    public function cheques()
    {
        return $this->hasMany(Cheque::class);
    }

    /**
     * Get the bank reconciliations for the bank account.
     */
    public function bankReconciliations()
    {
        return $this->hasMany(BankReconciliation::class);
    }

    /**
     * Alias for bankTransactions() to maintain compatibility
     */
    public function transactions()
    {
        return $this->bankTransactions();
    }

    /**
     * Calculate the balance as of a specific date
     * 
     * @param string $date The date to calculate the balance as of
     * @return float The balance as of the specified date
     */
    public function getBalanceAsOf($date)
    {
        // Start with opening balance
        $balance = $this->opening_balance;

        // If the date is before the opening date, return the opening balance
        if (strtotime($date) <= strtotime($this->opening_date)) {
            return $balance;
        }

        // Add all transactions up to the specified date
        $transactions = $this->bankTransactions()
            ->where('transaction_date', '<', $date)
            ->where('status', 'completed')
            ->get();

        foreach ($transactions as $transaction) {
            if ($transaction->transaction_type == 'deposit') {
                $balance += $transaction->amount;
            } else {
                $balance -= $transaction->amount;
            }
        }

        return $balance;
    }
}
