<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Support\Facades\Storage;

class Attachment extends Model
{
    use HasFactory, SoftDeletes;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'attachable_id',
        'attachable_type',
        'file_path',
        'file_name',
        'file_size',
        'file_type',
        'description',
        'is_public',
        'uploaded_by',
        'created_by',
        'updated_by',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'file_size' => 'integer',
        'is_public' => 'boolean',
    ];

    /**
     * Get the parent model (polymorphic).
     */
    public function attachable()
    {
        return $this->morphTo();
    }

    /**
     * Get the user who uploaded the attachment.
     */
    public function uploader()
    {
        return $this->belongsTo(User::class, 'uploaded_by');
    }

    /**
     * Get the user who created the attachment.
     */
    public function creator()
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    /**
     * Get the user who last updated the attachment.
     */
    public function updater()
    {
        return $this->belongsTo(User::class, 'updated_by');
    }

    /**
     * Get the file URL.
     *
     * @return string
     */
    public function getFileUrl()
    {
        return Storage::url($this->file_path);
    }

    /**
     * Get the formatted file size.
     *
     * @return string
     */
    public function getFormattedFileSizeAttribute()
    {
        $size = $this->file_size;

        if ($size < 1024) {
            return $size . ' بايت';
        } elseif ($size < 1048576) {
            return round($size / 1024, 2) . ' كيلوبايت';
        } elseif ($size < 1073741824) {
            return round($size / 1048576, 2) . ' ميجابايت';
        } else {
            return round($size / 1073741824, 2) . ' جيجابايت';
        }
    }

    /**
     * Get the file extension.
     *
     * @return string
     */
    public function getFileExtensionAttribute()
    {
        return pathinfo($this->file_name, PATHINFO_EXTENSION);
    }

    /**
     * Check if the file is an image.
     *
     * @return bool
     */
    public function isImage()
    {
        $imageExtensions = ['jpg', 'jpeg', 'png', 'gif', 'bmp', 'svg', 'webp'];
        return in_array(strtolower($this->getFileExtensionAttribute()), $imageExtensions);
    }

    /**
     * Check if the file is a PDF.
     *
     * @return bool
     */
    public function isPdf()
    {
        return strtolower($this->getFileExtensionAttribute()) === 'pdf';
    }

    /**
     * Check if the file is an office document.
     *
     * @return bool
     */
    public function isOfficeDocument()
    {
        $officeExtensions = ['doc', 'docx', 'xls', 'xlsx', 'ppt', 'pptx'];
        return in_array(strtolower($this->getFileExtensionAttribute()), $officeExtensions);
    }

    /**
     * Get the file icon based on file type.
     *
     * @return string
     */
    public function getFileIconAttribute()
    {
        $extension = $this->getFileExtensionAttribute();

        switch (strtolower($extension)) {
            case 'pdf':
                return 'fa-file-pdf';
            case 'doc':
            case 'docx':
                return 'fa-file-word';
            case 'xls':
            case 'xlsx':
                return 'fa-file-excel';
            case 'ppt':
            case 'pptx':
                return 'fa-file-powerpoint';
            case 'jpg':
            case 'jpeg':
            case 'png':
            case 'gif':
            case 'bmp':
                return 'fa-file-image';
            case 'zip':
            case 'rar':
            case '7z':
                return 'fa-file-archive';
            case 'txt':
                return 'fa-file-alt';
            case 'mp4':
            case 'avi':
            case 'mov':
            case 'wmv':
                return 'fa-file-video';
            case 'mp3':
            case 'wav':
            case 'ogg':
                return 'fa-file-audio';
            case 'html':
            case 'htm':
            case 'css':
            case 'js':
            case 'php':
                return 'fa-file-code';
            default:
                return 'fa-file';
        }
    }
}
