<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class AccountingPeriod extends Model
{
    use HasFactory, SoftDeletes;

    /**
     * الحقول القابلة للتعبئة الجماعية
     *
     * @var array
     */
    protected $fillable = [
        'fiscal_year_id',
        'company_id',
        'name',
        'start_date',
        'end_date',
        'is_closed'
    ];

    /**
     * الحقول التي يجب تحويلها إلى أنواع محددة
     *
     * @var array
     */
    protected $casts = [
        'start_date' => 'date',
        'end_date' => 'date',
        'is_closed' => 'boolean'
    ];

    /**
     * علاقة مع السنة المالية
     */
    public function fiscalYear()
    {
        return $this->belongsTo(FiscalYear::class);
    }

    /**
     * علاقة مع الشركة
     */
    public function company()
    {
        return $this->belongsTo(Company::class);
    }

    /**
     * علاقة مع القيود المحاسبية
     */
    public function journalEntries()
    {
        return $this->hasMany(JournalEntry::class);
    }

    /**
     * نطاق للفترات المحاسبية المفتوحة (غير مغلقة)
     */
    public function scopeOpen($query)
    {
        return $query->where('is_closed', false);
    }

    /**
     * نطاق للفترات المحاسبية المغلقة
     */
    public function scopeClosed($query)
    {
        return $query->where('is_closed', true);
    }

    /**
     * التحقق مما إذا كان التاريخ ضمن الفترة المحاسبية
     */
    public function isDateWithinPeriod($date)
    {
        $checkDate = $date instanceof \DateTime ? $date : new \DateTime($date);
        $startDate = new \DateTime($this->start_date);
        $endDate = new \DateTime($this->end_date);

        return $checkDate >= $startDate && $checkDate <= $endDate;
    }

    /**
     * الحصول على إجمالي الإيرادات للفترة
     */
    public function getTotalRevenue()
    {
        return $this->journalEntries()
            ->with('items.account')
            ->whereHas('items.account', function ($query) {
                $query->where('account_type', 'revenue');
            })
            ->get()
            ->sum(function ($entry) {
                return $entry->items->where('account.account_type', 'revenue')->sum('credit');
            });
    }

    /**
     * الحصول على إجمالي المصروفات للفترة
     */
    public function getTotalExpenses()
    {
        return $this->journalEntries()
            ->with('items.account')
            ->whereHas('items.account', function ($query) {
                $query->where('account_type', 'expense');
            })
            ->get()
            ->sum(function ($entry) {
                return $entry->items->where('account.account_type', 'expense')->sum('debit');
            });
    }

    /**
     * الحصول على صافي الربح للفترة
     */
    public function getNetProfit()
    {
        return $this->getTotalRevenue() - $this->getTotalExpenses();
    }

    /**
     * إغلاق الفترة المحاسبية
     */
    public function close()
    {
        // التحقق من أن جميع القيود المحاسبية مرحلة ومعتمدة
        $pendingEntries = $this->journalEntries()
            ->where(function ($query) {
                $query->where('is_posted', false)
                    ->orWhere('is_approved', false);
            })
            ->count();

        if ($pendingEntries > 0) {
            throw new \Exception('لا يمكن إغلاق الفترة المحاسبية لوجود قيود غير مرحلة أو غير معتمدة');
        }

        $this->is_closed = true;
        $this->save();

        // التحقق من إغلاق جميع الفترات في السنة المالية
        $openPeriods = $this->fiscalYear->accountingPeriods()->where('is_closed', false)->count();

        if ($openPeriods == 0) {
            // إغلاق السنة المالية إذا تم إغلاق جميع الفترات
            $this->fiscalYear->is_closed = true;
            $this->fiscalYear->save();
        }

        return true;
    }
}
