<?php

namespace App\Listeners;

use Illuminate\Auth\Events\Login;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Session;
use App\Models\JournalEntry;
use App\Models\ChartOfAccount;

class CheckTrialBalanceOnLogin
{
    /**
     * إنشاء المستمع
     *
     * @return void
     */
    public function __construct()
    {
        //
    }

    /**
     * معالجة الحدث
     *
     * @param  Login  $event
     * @return void
     */
    public function handle(Login $event)
    {
        // التحقق فقط للمستخدمين الذين لديهم صلاحية إدارة الحسابات
        if ($event->user->can('manage_accounting')) {
            try {
                // فحص توازن قيود اليومية
                $unbalancedEntries = $this->checkJournalEntriesBalance();

                // فحص أرصدة الحسابات
                $unbalancedAccounts = $this->checkAccountsBalance();

                // إذا كان هناك قيود غير متوازنة أو حسابات غير متوازنة، نعرض رسالة للمستخدم
                if (count($unbalancedEntries) > 0 || count($unbalancedAccounts) > 0) {
                    $message = 'تم اكتشاف مشكلة في ميزان المراجعة: ';

                    if (count($unbalancedEntries) > 0) {
                        $message .= count($unbalancedEntries) . ' قيد غير متوازن';
                    }

                    if (count($unbalancedEntries) > 0 && count($unbalancedAccounts) > 0) {
                        $message .= ' و ';
                    }

                    if (count($unbalancedAccounts) > 0) {
                        $message .= count($unbalancedAccounts) . ' حساب غير متوازن';
                    }

                    $message .= '. يرجى تشغيل أداة فحص ميزان المراجعة لتصحيح المشكلة.';

                    Session::flash('warning', $message);

                    // تسجيل المشكلة في ملف السجل
                    Log::warning('تم اكتشاف مشكلة في ميزان المراجعة عند تسجيل دخول المستخدم', [
                        'user_id' => $event->user->id,
                        'unbalanced_entries_count' => count($unbalancedEntries),
                        'unbalanced_accounts_count' => count($unbalancedAccounts)
                    ]);
                }
            } catch (\Exception $e) {
                // تسجيل الخطأ في ملف السجل
                Log::error('خطأ في فحص ميزان المراجعة عند تسجيل الدخول', [
                    'user_id' => $event->user->id,
                    'error' => $e->getMessage()
                ]);
            }
        }
    }

    /**
     * فحص توازن قيود اليومية
     *
     * @return array
     */
    private function checkJournalEntriesBalance()
    {
        $unbalancedEntries = [];

        // الحصول على معرف الشركة للمستخدم الحالي
        $companyId = auth()->user()->company_id;

        // تطبيق عزل الشركات - فقط فحص القيود الخاصة بشركة المستخدم
        $entries = JournalEntry::where('company_id', $companyId)->get();

        foreach ($entries as $entry) {
            if (!$entry->isBalanced()) {
                $difference = $entry->total_debit - $entry->total_credit;

                $unbalancedEntries[] = [
                    'id' => $entry->id,
                    'entry_number' => $entry->entry_number,
                    'entry_date' => $entry->entry_date,
                    'description' => $entry->description,
                    'total_debit' => $entry->total_debit,
                    'total_credit' => $entry->total_credit,
                    'difference' => $difference
                ];
            }
        }

        return $unbalancedEntries;
    }

    /**
     * فحص أرصدة الحسابات
     *
     * @return array
     */
    private function checkAccountsBalance()
    {
        $unbalancedAccounts = [];

        // الحصول على معرف الشركة للمستخدم الحالي
        $companyId = auth()->user()->company_id;

        // تطبيق عزل الشركات - فقط فحص الحسابات الخاصة بشركة المستخدم
        $accounts = ChartOfAccount::where('company_id', $companyId)->get();

        foreach ($accounts as $account) {
            // حساب الرصيد من بنود القيود المحاسبية
            // تطبيق عزل الشركات على بنود القيود المحاسبية أيضًا
            $journalItems = $account->journalItems()
                ->join('journal_entries', 'journal_entry_items.journal_entry_id', '=', 'journal_entries.id')
                ->where('journal_entries.company_id', $companyId);

            $debits = $journalItems->sum('journal_entry_items.debit');
            $credits = $journalItems->sum('journal_entry_items.credit');

            // حساب الرصيد المتوقع
            $expectedBalance = 0;
            if (in_array($account->account_type, ['asset', 'expense'])) {
                $expectedBalance = $account->opening_balance + $debits - $credits;
            } else {
                $expectedBalance = $account->opening_balance + $credits - $debits;
            }

            // مقارنة الرصيد المتوقع بالرصيد الحالي
            if (abs($expectedBalance - $account->current_balance) > 0.01) {
                $unbalancedAccounts[] = [
                    'id' => $account->id,
                    'name' => $account->name,
                    'account_code' => $account->account_code,
                    'current_balance' => $account->current_balance,
                    'expected_balance' => $expectedBalance,
                    'difference' => $expectedBalance - $account->current_balance
                ];
            }
        }

        return $unbalancedAccounts;
    }
}
