<?php

namespace App\Imports;

use App\Models\Elevator;
use App\Models\Location;
use App\Models\Customer;
use Illuminate\Support\Collection;
use Maatwebsite\Excel\Concerns\ToCollection;
use Maatwebsite\Excel\Concerns\WithHeadingRow;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;

class ElevatorsImport implements ToCollection, WithHeadingRow
{
    public $importedCount = 0;
    public $skippedCount = 0;
    public $duplicateCount = 0;
    public $errors = [];

    public function collection(Collection $rows)
    {
        $user = Auth::user();
        $isSystemAdmin = $user->isSystemAdmin();
        
        Log::info("Starting collection processing. Total rows received: " . $rows->count());

        if ($rows->isEmpty()) {
            $this->errors[] = "الملف فارغ أو لم يتم العثور على بيانات في الصفوف.";
            Log::warning("Import failed: Empty collection.");
            return;
        }

        foreach ($rows as $index => $row) {
            $rowNumber = $index + 2; 
            Log::info("Processing row {$rowNumber}: " . json_encode($row->toArray()));

            // 1. Manual Validation for the row
            $validator = Validator::make($row->toArray(), $this->getRowRules($user));

            if ($validator->fails()) {
                $this->skippedCount++;
                $errorString = implode(', ', $validator->errors()->all());
                $this->errors[] = "الصف {$rowNumber}: " . $errorString;
                Log::warning("Row {$rowNumber} failed validation: {$errorString}");
                continue;
            }

            // 2. Determine company_id
            $company_id = ($isSystemAdmin && isset($row['company_id'])) ? $row['company_id'] : $user->company_id;

            if (!$company_id) {
                $this->skippedCount++;
                $this->errors[] = "الصف {$rowNumber}: لم يتم تحديد الشركة (company_id).";
                Log::warning("Row {$rowNumber} skipped: No company_id found.");
                continue;
            }

            // 3. Customer Lookup
            $customerId = $row['customer_id'] ?? null;
            if (empty($customerId) && !empty($row['customer_name'])) {
                $customer = Customer::where('name', $row['customer_name'])
                    ->where('company_id', $company_id)
                    ->first();
                if ($customer) {
                    $customerId = $customer->id;
                } else {
                    $this->errors[] = "الصف {$rowNumber}: لم يتم العثور على عميل باسم: " . $row['customer_name'];
                    Log::warning("Row {$rowNumber}: Customer name not found in DB.");
                }
            }

            if (empty($customerId)) {
                $this->skippedCount++;
                $this->errors[] = "الصف {$rowNumber}: معرف العميل مفقود أو غير صالح.";
                Log::warning("Row {$rowNumber} skipped: Customer ID missing.");
                continue;
            }

            // 4. Duplicate Check
            $existingElevator = Elevator::where('serial_number', $row['serial_number'])
                ->where('company_id', $company_id)
                ->first();

            if ($existingElevator) {
                $this->duplicateCount++;
                Log::info("Row {$rowNumber} skipped: Duplicate serial number.");
                continue;
            }

            // 5. Location Lookup
            $locationId = $row['location_id'] ?? null;
            if (empty($locationId) && !empty($row['location_name'])) {
                $location = Location::where('name', $row['location_name'])
                    ->where('company_id', $company_id)
                    ->where('customer_id', $customerId)
                    ->first();
                if ($location) {
                    $locationId = $location->id;
                }
            }

            if (empty($locationId)) {
                $defaultLocation = Location::where('name', 'موقع افتراضي')
                    ->where('company_id', $company_id)
                    ->where('customer_id', $customerId)
                    ->first();

                if (!$defaultLocation) {
                    try {
                        $defaultLocation = Location::create([
                            'company_id' => $company_id,
                            'customer_id' => $customerId,
                            'name' => 'موقع افتراضي',
                            'address' => 'عنوان افتراضي',
                            'city' => 'المدينة',
                            'state' => 'المنطقة',
                            'country' => 'البلد',
                            'postal_code' => '00000',
                            'contact_person' => 'غير محدد',
                            'contact_phone' => 'غير محدد'
                        ]);
                    } catch (\Exception $e) {
                        $this->skippedCount++;
                        $this->errors[] = "الصف {$rowNumber}: فشل إنشاء موقع افتراضي: " . $e->getMessage();
                        continue;
                    }
                }
                $locationId = $defaultLocation->id;
            }

            // 6. Final Creation
            try {
                Elevator::create([
                    'company_id' => $company_id,
                    'customer_id' => $customerId,
                    'location_id' => $locationId,
                    'user_id' => $user->id,
                    'branch_id' => $user->branch_id,
                    'serial_number' => $row['serial_number'],
                    'model' => $row['model'],
                    'manufacturer' => $row['manufacturer'],
                    'installation_date' => $row['installation_date'],
                    'capacity' => $row['capacity'],
                    'floors_count' => $row['floors_count'],
                    'speed' => $row['speed'],
                    'price' => $row['price'] ?? null,
                    'type' => $row['type'] ?? 'passenger',
                    'status' => $row['status'] ?? 'operational',
                    'warranty_end_date' => $row['warranty_end_date'] ?? null,
                    'free_maintenance_end_date' => $row['free_maintenance_end_date'] ?? null,
                    'control_warranty_end_date' => $row['control_warranty_end_date'] ?? null,
                    'machine_warranty_end_date' => $row['machine_warranty_end_date'] ?? null,
                    'delivery_date' => $row['delivery_date'] ?? null,
                    'specifications' => $row['specifications'] ?? null,
                    'notes' => $row['notes'] ?? null,
                ]);

                $this->importedCount++;
                Log::info("Row {$rowNumber} imported successfully.");
            } catch (\Exception $e) {
                $this->skippedCount++;
                $this->errors[] = "الصف {$rowNumber}: خطأ في الحفظ: " . $e->getMessage();
                Log::error("Row {$rowNumber} DB error: " . $e->getMessage());
            }
        }
        
        Log::info("Finished collection processing. Imported: {$this->importedCount}, Skipped: {$this->skippedCount}");
    }

    private function getRowRules($user)
    {
        $rules = [
            'serial_number' => 'required',
            'model' => 'nullable',
            'manufacturer' => 'nullable',
            'installation_date' => 'nullable',
            'capacity' => 'nullable|numeric',
            'floors_count' => 'nullable|integer',
            'speed' => 'nullable|numeric',
            'price' => 'nullable|numeric',
            'type' => 'nullable|in:passenger,freight,hospital,panoramic,residential,commercial,other',
            'status' => 'nullable|in:operational,under_maintenance,out_of_service,installation',
        ];

        if ($user->isSystemAdmin()) {
            $rules['company_id'] = 'required_without_all:customer_id,customer_name';
        }

        return $rules;
    }
}
