<?php

namespace App\Http\Controllers;

use App\Models\Invoice;
use App\Services\ZatcaService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class ZatcaController extends Controller
{
    protected $zatcaService;

    public function __construct(ZatcaService $zatcaService)
    {
        $this->zatcaService = $zatcaService;
        $this->middleware('auth');

        // التحقق من الصلاحيات
        $this->middleware('permission:view_zatca_invoices')->only(['index', 'show']);
        $this->middleware('permission:manage_zatca_invoices')->only(['updateInvoiceType']);
        $this->middleware('permission:generate_zatca_qr_codes')->only(['generateQrCode']);
        $this->middleware('permission:generate_zatca_xml_files')->only(['generateXml']);
        $this->middleware('permission:report_zatca_invoices')->only(['reportInvoice']);
        $this->middleware('permission:check_zatca_invoice_status')->only(['checkStatus']);
    }

    /**
     * عرض صفحة إدارة الفوترة الإلكترونية
     *
     * @return \Illuminate\View\View
     */
    public function index()
    {
        $invoices = Invoice::whereNotNull('zatca_invoice_type')
            ->orWhere('zatca_reporting_status', '!=', null)
            ->orderBy('created_at', 'desc')
            ->paginate(15);

        return view('zatca.index', compact('invoices'));
    }

    /**
     * عرض تفاصيل فاتورة معينة
     *
     * @param  int  $id
     * @return \Illuminate\View\View
     */
    public function show($id)
    {
        $invoice = Invoice::findOrFail($id);
        $status = $this->zatcaService->checkInvoiceStatus($invoice);

        return view('zatca.show', compact('invoice', 'status'));
    }

    /**
     * تحديث نوع الفاتورة
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\RedirectResponse
     */
    public function updateInvoiceType(Request $request, $id)
    {
        $invoice = Invoice::findOrFail($id);

        $request->validate([
            'zatca_invoice_type' => 'required|in:standard,simplified,debit_note,credit_note',
        ]);

        $invoice->zatca_invoice_type = $request->zatca_invoice_type;
        $invoice->save();

        return redirect()->route('zatca.show', $id)->with('success', 'تم تحديث نوع الفاتورة بنجاح');
    }

    /**
     * إنشاء رمز الاستجابة السريعة للفاتورة
     *
     * @param  int  $id
     * @return \Illuminate\Http\RedirectResponse
     */
    public function generateQrCode($id)
    {
        $invoice = Invoice::findOrFail($id);

        try {
            $qrCode = $this->zatcaService->generateQrCode($invoice);
            $invoice->zatca_qr_code = $qrCode;
            $invoice->save();

            return redirect()->route('zatca.show', $id)->with('success', 'تم إنشاء رمز الاستجابة السريعة بنجاح');
        } catch (\Exception $e) {
            Log::error('فشل في إنشاء رمز الاستجابة السريعة: ' . $e->getMessage());
            return redirect()->route('zatca.show', $id)->with('error', 'فشل في إنشاء رمز الاستجابة السريعة: ' . $e->getMessage());
        }
    }

    /**
     * إنشاء ملف XML للفاتورة
     *
     * @param  int  $id
     * @return \Illuminate\Http\RedirectResponse
     */
    public function generateXml($id)
    {
        $invoice = Invoice::findOrFail($id);

        try {
            $xmlPath = $this->zatcaService->generateXml($invoice);

            return redirect()->route('zatca.show', $id)->with('success', 'تم إنشاء ملف XML بنجاح');
        } catch (\Exception $e) {
            Log::error('فشل في إنشاء ملف XML: ' . $e->getMessage());
            return redirect()->route('zatca.show', $id)->with('error', 'فشل في إنشاء ملف XML: ' . $e->getMessage());
        }
    }

    /**
     * إرسال الفاتورة إلى هيئة الزكاة والدخل
     *
     * @param  int  $id
     * @return \Illuminate\Http\RedirectResponse
     */
    public function reportInvoice($id)
    {
        $invoice = Invoice::findOrFail($id);

        try {
            $result = $this->zatcaService->reportInvoice($invoice);

            if ($result) {
                return redirect()->route('zatca.show', $id)->with('success', 'تم إرسال الفاتورة بنجاح إلى هيئة الزكاة والدخل');
            } else {
                return redirect()->route('zatca.show', $id)->with('error', 'فشل في إرسال الفاتورة إلى هيئة الزكاة والدخل');
            }
        } catch (\Exception $e) {
            Log::error('فشل في إرسال الفاتورة إلى هيئة الزكاة والدخل: ' . $e->getMessage());
            return redirect()->route('zatca.show', $id)->with('error', 'فشل في إرسال الفاتورة إلى هيئة الزكاة والدخل: ' . $e->getMessage());
        }
    }

    /**
     * التحقق من حالة الفاتورة لدى هيئة الزكاة والدخل
     *
     * @param  int  $id
     * @return \Illuminate\Http\RedirectResponse
     */
    public function checkStatus($id)
    {
        $invoice = Invoice::findOrFail($id);

        try {
            $status = $this->zatcaService->checkInvoiceStatus($invoice);

            return redirect()->route('zatca.show', $id)->with('status', $status);
        } catch (\Exception $e) {
            Log::error('فشل في التحقق من حالة الفاتورة: ' . $e->getMessage());
            return redirect()->route('zatca.show', $id)->with('error', 'فشل في التحقق من حالة الفاتورة: ' . $e->getMessage());
        }
    }
}
