<?php

namespace App\Http\Controllers;

use App\Models\Elevator;
use App\Models\ElevatorStage;
use App\Models\InstallationProject;
use App\Models\ProjectPhase;
use App\Models\User;
use App\Models\WorkOrder;
use App\Exports\WorkOrdersExport;
use Maatwebsite\Excel\Facades\Excel;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class WorkOrderController extends Controller
{
    /**
     * إنشاء مثيل جديد من وحدة التحكم.
     *
     * @return void
     */
    public function __construct()
    {
        $this->middleware('permission:view_work_orders', ['only' => ['index', 'show', 'getElevatorDetails']]);
        $this->middleware('permission:create_work_orders', ['only' => ['create', 'store']]);
        $this->middleware('permission:edit_work_orders', ['only' => ['edit', 'update']]);
        $this->middleware('permission:delete_work_orders', ['only' => ['destroy']]);
    }
    /**
     * عرض قائمة أوامر الشغل
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        $companyId = Auth::user()->company_id;
        $query = WorkOrder::where('company_id', $companyId)
            ->with(['elevator.customer', 'projectPhase', 'elevatorStage', 'technician'])
            ->orderBy('created_at', 'desc');

        if ($request->filled('elevator_number')) {
            $query->whereHas('elevator', function ($q) use ($request) {
                $q->where('serial_number', 'like', '%' . $request->elevator_number . '%')
                  ->orWhere('model', 'like', '%' . $request->elevator_number . '%');
            });
        }

        if ($request->filled('customer_name')) {
            $query->whereHas('elevator.customer', function ($q) use ($request) {
                $q->where('name', 'like', '%' . $request->customer_name . '%');
            });
        }

        if ($request->filled('stage_name')) {
            $query->where('elevator_stage_id', $request->stage_name);
        }

        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        $workOrders = $query->paginate(10)->appends($request->all());
        $elevatorStages = ElevatorStage::where('company_id', $companyId)->get();

        return view('work_orders.index', compact('workOrders', 'elevatorStages'));
    }

    /**
     * تصدير أوامر الشغل إلى ملف إكسيل
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Symfony\Component\HttpFoundation\BinaryFileResponse
     */
    public function export(Request $request)
    {
        return Excel::download(new WorkOrdersExport($request), 'work_orders_' . date('Y-m-d') . '.xlsx');
    }

    /**
     * عرض نموذج إنشاء أمر شغل جديد
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $companyId = Auth::user()->company_id;

        // الحصول على المصاعد التي حالتها تحت التركيب
        $elevators = Elevator::where('company_id', $companyId)
            ->where('status', 'installation')
            ->with('customer', 'location')
            ->get();

        // الحصول على مراحل المصاعد
        $elevatorStages = ElevatorStage::where('company_id', $companyId)
            ->where('is_active', true)
            ->orderBy('order')
            ->get();

        // الحصول على الفنيين (المستخدمين الذين لديهم أي دور متعلق بالفنيين)
        $technicians = User::where('company_id', $companyId)
            ->where('is_active', true)
            ->whereHas('roles', function ($query) {
                $query->whereIn('slug', ['technician', 'maintenance_technician'])
                    ->orWhereIn('name', ['فني', 'مدير الصيانة', 'فني تركيب', 'فني صيانة']);
            })
            ->get();

        return view('work_orders.create', compact('elevators', 'elevatorStages', 'technicians'));
    }

    /**
     * تخزين أمر شغل جديد
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $request->validate([
            'elevator_id' => 'required|exists:elevators,id',
            'elevator_stage_id' => 'required|exists:elevator_stages,id',
            'technician_id' => 'nullable|exists:users,id',
            'status' => 'required|in:open,paused,completed,archived',
            'is_technically_ready' => 'required|boolean',
            'order_date' => 'required|date',
            'received_date' => 'nullable|date',
            'completion_date' => 'nullable|date',
            'notes' => 'nullable|string'
        ]);

        // التحقق من كفاية المدفوعات للمرحلة المختارة
        $elevator = Elevator::findOrFail($request->elevator_id);

        // حساب قيمة التعاقد (من مشروع التركيب أو سعر المصعد)
        $contractValue = 0;
        if ($elevator->installationProject) {
            $contractValue = $elevator->installationProject->contract_value ?? 0;
        }

        // استخدام سعر المصعد إذا كان متاحاً وقيمة التعاقد صفر
        if ($contractValue <= 0) {
            $contractValue = $elevator->price ?? 0;
        }

        // حساب إجمالي المبالغ المدفوعة من الخزينة
        $cashPayments = \App\Models\CashTransaction::where('elevator_id', $request->elevator_id)
            ->where('type', 'deposit')
            ->sum('amount');

        // حساب إجمالي المبالغ المدفوعة من البنوك
        $bankPayments = \App\Models\BankTransaction::where('transaction_type', 'deposit')
            ->where(function ($query) use ($request) {
                $query->whereJsonContains('additional_data->elevator_id', $request->elevator_id)
                    ->orWhereJsonContains('additional_data->elevator_id', (string)$request->elevator_id);
            })
            ->sum('amount');

        // إجمالي المبالغ المدفوعة
        $totalPaid = $cashPayments + $bankPayments;

        // حساب نسبة المدفوع من قيمة التعاقد
        $paymentPercentage = 0;
        if ($contractValue > 0) {
            $paymentPercentage = ($totalPaid / $contractValue) * 100;
        }

        // التحقق من كفاية المدفوعات بناءً على مرحلة المصعد المختارة
        $elevatorStage = ElevatorStage::findOrFail($request->elevator_stage_id);
        $requiredPercentage = $elevatorStage->percentage ?? 0;

        // تم إزالة التحقق الإجباري من كفاية المدفوعات للسماح بمرونة أكثر في النظام
        // وسيتم الاكتفاء بالتنبيه في واجهة المستخدم
        /*
        if ($paymentPercentage < $requiredPercentage) {
            return redirect()->back()->withInput()
                ->with('error', 'لا يمكن إنشاء أمر الشغل. مدفوعات العميل غير كافية للمرحلة المختارة. النسبة المطلوبة: ' . $requiredPercentage . '% والنسبة المدفوعة: ' . number_format($paymentPercentage, 2) . '%');
        }
        */

        // التحقق من عدم وجود أمر شغل سابق لنفس المصعد ونفس المرحلة
        $existingWorkOrder = WorkOrder::where('elevator_id', $request->elevator_id)
            ->where('elevator_stage_id', $request->elevator_stage_id)
            ->whereNull('deleted_at') // للتأكد من عدم احتساب الأوامر المحذوفة
            ->first();

        if ($existingWorkOrder) {
            return redirect()->back()->withInput()
                ->with('error', 'لا يمكن إنشاء أمر الشغل. يوجد بالفعل أمر شغل للمصعد في مرحلة "' . $elevatorStage->name . '".');
        }

        $workOrder = new WorkOrder();
        $workOrder->elevator_id = $request->elevator_id;
        $workOrder->elevator_stage_id = $request->elevator_stage_id;
        $workOrder->project_phase_id = null; // لا نستخدم مرحلة المشروع

        $workOrder->technician_id = $request->technician_id; // يمكن أن يكون null
        $workOrder->company_id = Auth::user()->company_id;
        $workOrder->status = $request->status;
        $workOrder->is_technically_ready = $request->is_technically_ready;
        $workOrder->order_date = $request->order_date;
        $workOrder->received_date = $request->received_date;
        $workOrder->completion_date = $request->completion_date;
        $workOrder->notes = $request->notes;
        $workOrder->created_by = Auth::id();
        $workOrder->save();

        return redirect()->route('work-orders.index')
            ->with('success', 'تم إنشاء أمر الشغل بنجاح');
    }

    /**
     * عرض تفاصيل أمر شغل محدد
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $companyId = Auth::user()->company_id;
        $workOrder = WorkOrder::where('company_id', $companyId)
            ->where('id', $id)
            ->with(['elevator.customer', 'elevator.location', 'projectPhase', 'elevatorStage', 'technician', 'creator'])
            ->firstOrFail();

        return view('work_orders.show', compact('workOrder'));
    }

    /**
     * عرض نموذج تعديل أمر شغل
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $companyId = Auth::user()->company_id;
        $workOrder = WorkOrder::where('company_id', $companyId)
            ->where('id', $id)
            ->with(['projectPhase', 'elevator.customer', 'elevator.location', 'elevator.installationProject']) // تحميل العلاقات المطلوبة
            ->firstOrFail();

        // الحصول على المصاعد التي حالتها تحت التركيب
        $elevators = Elevator::where('company_id', $companyId)
            ->where('status', 'installation')
            ->with('customer', 'location')
            ->get();

        // الحصول على مراحل المصاعد
        $elevatorStages = ElevatorStage::where('company_id', $companyId)
            ->where('is_active', true)
            ->orderBy('order')
            ->get();

        // لا نحتاج إلى تحميل مراحل المشروع لأننا لن نستخدمها

        // الحصول على الفنيين (المستخدمين الذين لديهم أي دور متعلق بالفنيين)
        $technicians = User::where('company_id', $companyId)
            ->where('is_active', true)
            ->whereHas('roles', function ($query) {
                $query->whereIn('slug', ['technician', 'maintenance_technician'])
                    ->orWhereIn('name', ['فني', 'مدير الصيانة', 'فني تركيب', 'فني صيانة']);
            })
            ->get();

        return view('work_orders.edit', compact('workOrder', 'elevators', 'elevatorStages', 'technicians'));
    }

    /**
     * تحديث أمر شغل محدد
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $request->validate([
            'elevator_id' => 'required|exists:elevators,id',
            // لا نتحقق من elevator_stage_id لأنه أصبح للقراءة فقط
            'technician_id' => 'nullable|exists:users,id',
            'status' => 'required|in:open,paused,completed,archived',
            'is_technically_ready' => 'required|boolean',
            'order_date' => 'required|date',
            'received_date' => 'nullable|date',
            'completion_date' => 'nullable|date',
            'notes' => 'nullable|string'
        ]);

        // لا نحتاج للتحقق من وجود مرحلة مشروع أو مرحلة مصعد لأن هذه الحقول أصبحت للقراءة فقط
        // وسيتم استخدام القيم الموجودة مسبقاً

        $companyId = Auth::user()->company_id;
        $workOrder = WorkOrder::where('company_id', $companyId)
            ->where('id', $id)
            ->firstOrFail();

        $workOrder->elevator_id = $request->elevator_id;
        $workOrder->project_phase_id = null; // لا نستخدم مرحلة المشروع
        // لا نقوم بتحديث elevator_stage_id لأنه أصبح للقراءة فقط
        $workOrder->technician_id = $request->technician_id;
        $workOrder->status = $request->status;
        $workOrder->is_technically_ready = $request->is_technically_ready;
        $workOrder->order_date = $request->order_date;
        $workOrder->received_date = $request->received_date;
        $workOrder->completion_date = $request->completion_date;
        $workOrder->notes = $request->notes;
        $workOrder->save();

        return redirect()->route('work-orders.index')
            ->with('success', 'تم تحديث أمر الشغل بنجاح');
    }

    /**
     * حذف أمر شغل محدد
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        $companyId = Auth::user()->company_id;
        $workOrder = WorkOrder::where('company_id', $companyId)
            ->where('id', $id)
            ->firstOrFail();

        $workOrder->delete();

        return redirect()->route('work-orders.index')
            ->with('success', 'تم حذف أمر الشغل بنجاح');
    }

    /**
     * الحصول على بيانات المصعد بناءً على المعرف
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function getElevatorDetails($id)
    {
        try {
            $companyId = Auth::user()->company_id;
            $elevator = Elevator::where('company_id', $companyId)
                ->where('id', $id)
                ->with(['customer', 'location', 'installationProject'])
                ->firstOrFail();

            // التحقق من وجود العميل والموقع
            $customerName = $elevator->customer ? $elevator->customer->name : 'غير محدد';

            $locationData = [
                'id' => null,
                'name' => 'غير محدد',
                'address' => ''
            ];

            if ($elevator->location) {
                $locationData = [
                    'id' => $elevator->location->id,
                    'name' => $elevator->location->name,
                    'address' => $elevator->location->address ?? ''
                ];
            }

            // التحقق من وجود مشروع التركيب
            $contractValue = 0;
            if ($elevator->installationProject) {
                $contractValue = $elevator->installationProject->contract_value ?? 0;
            }

            // استخدام سعر المصعد إذا كان متاحاً
            $elevatorPrice = $elevator->price ?? 0;

            // حساب إجمالي المبالغ المدفوعة من الخزينة
            $cashPayments = \App\Models\CashTransaction::where('elevator_id', $id)
                ->where('type', 'deposit')
                ->sum('amount');

            // حساب إجمالي المبالغ المدفوعة من البنوك
            $bankPayments = \App\Models\BankTransaction::where('transaction_type', 'deposit')
                ->where(function ($query) use ($id) {
                    $query->whereJsonContains('additional_data->elevator_id', $id)
                        ->orWhereJsonContains('additional_data->elevator_id', (string)$id);
                })
                ->sum('amount');

            // إجمالي المبالغ المدفوعة
            $totalPaid = $cashPayments + $bankPayments;

            return response()->json([
                'customer' => $customerName,
                'location' => $locationData,
                'contract_value' => $contractValue,
                'elevator_price' => $elevatorPrice,
                'total_paid' => $totalPaid
            ]);
        } catch (\Exception $e) {
            // تسجيل الخطأ للمساعدة في تشخيص المشكلة
            \Log::error('خطأ في استرجاع بيانات المصعد: ' . $e->getMessage());

            return response()->json([
                'error' => 'حدث خطأ أثناء استرجاع بيانات المصعد: ' . $e->getMessage()
            ], 500);
        }
    }
}
