<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\User;
use App\Models\Role;
use App\Models\Branch;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rules\Password;
use Illuminate\Support\Facades\Schema;

class UserController extends Controller
{
    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct()
    {
        $this->middleware('auth');
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        $user = Auth::user();
        $company = $user->company;

        // إذا كان المستخدم مرتبطاً بشركة، نعرض فقط مستخدمي تلك الشركة
        // وإلا نعرض جميع المستخدمين (للمستخدم التجريبي أو مدير النظام)
        if ($user->isSystemAdmin()) {
            $query = User::query(); // عرض جميع المستخدمين
        } elseif ($company) {
            $query = User::where('company_id', $company->id);
        } else {
            $query = User::where('id', $user->id); // fallback
        }

        // Filter by branch if specified
        if ($request->has('branch_id') && $request->branch_id) {
            $query->where('branch_id', $request->branch_id);
            $branch = Branch::findOrFail($request->branch_id);
        }

        // Filter by role if specified
        if ($request->has('role') && $request->role) {
            $query->whereHas('roles', function ($q) use ($request) {
                $q->where('name', $request->role);
            });
        }

        $users = $query->orderBy('name')->paginate(10);

        // Get all branches for filter dropdown
        if ($user->isSystemAdmin()) {
            $branches = Branch::where('is_active', true)
                ->orderBy('name')
                ->get();
        } elseif ($company) {
            $branches = Branch::where('company_id', $company->id)
                ->where('is_active', true)
                ->orderBy('name')
                ->get();
        } else {
            $branches = collect();
        }

        // Get all roles for filter dropdown
        if ($user->isSystemAdmin()) {
            $roles = Role::orderBy('name')->get();
        } elseif ($company) {
            $roles = Role::whereIn('id', function ($query) use ($company) {
                $query->select('role_id')
                    ->from('role_user')
                    ->join('users', 'users.id', '=', 'role_user.user_id')
                    ->where('users.company_id', $company->id);
            })
                ->orderBy('name')
                ->get();
        } else {
            $roles = collect();
        }

        return view('users.index', compact('users', 'company', 'branches', 'roles'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $user = Auth::user();
        $company = $user->company;

        // تعديل: السماح للمستخدم بإنشاء مستخدمين جدد حتى لو لم يكن مرتبطًا بشركة
        // إذا كان لديه دور super_admin أو admin
        if (!$company && !$user->isSystemAdmin()) {
            return redirect()->route('companies.index')
                ->with('error', 'You need to be associated with a company first.');
        }

        // Check if user has permission to create users
        // تعديل: التحقق من الصلاحيات بالصيغتين (المفرد والجمع)
        if (!$user->hasPermissionTo('create_user') && !$user->hasPermissionTo('create_users')) {
            return redirect()->route('users.index')
                ->with('error', 'You do not have permission to create users.');
        }

        // Get all branches for dropdown
        if ($company) {
            // إذا كان المستخدم مرتبطًا بشركة، نعرض فقط فروع تلك الشركة
            $branches = Branch::where('company_id', $company->id)
                ->where('is_active', true)
                ->orderBy('name')
                ->get();
        } else {
            // إذا كان المستخدم super_admin وليس مرتبطًا بشركة، نعرض جميع الفروع
            $branches = Branch::where('is_active', true)
                ->orderBy('name')
                ->get();
        }

        // Get all roles for dropdown
        $roles = Role::where(function ($query) use ($user) {
            // Super admin can assign any role
            if ($user->hasRole('super_admin')) {
                return $query;
            }

            // Admin can assign any role except super_admin
            if ($user->hasRole('admin')) {
                return $query->where('name', '!=', 'super_admin');
            }

            // Other users can only assign basic roles
            return $query->whereIn('name', ['user', 'technician', 'accountant']);
        })
            ->orderBy('name')
            ->get();

        return view('users.create', compact('company', 'branches', 'roles'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        // عرض رسالة تأكيد لبدء عملية إنشاء المستخدم
        session()->flash('info', 'بدء عملية إنشاء المستخدم الجديد...');

        $user = Auth::user();
        $company = $user->company;

        // تعديل: السماح للمستخدم بإنشاء مستخدمين جدد حتى لو لم يكن مرتبطًا بشركة
        // إذا كان لديه دور super_admin أو admin
        if (!$company && !$user->isSystemAdmin()) {
            return redirect()->route('companies.index')
                ->with('error', 'يجب أن تكون مرتبطًا بشركة أولاً.');
        }

        // Check if user has permission to create users
        // تعديل: التحقق من الصلاحيات بالصيغتين (المفرد والجمع)
        if (!$user->hasPermissionTo('create_user') && !$user->hasPermissionTo('create_users')) {
            return redirect()->route('users.index')
                ->with('error', 'ليس لديك صلاحية لإنشاء مستخدمين جدد.');
        }

        // عرض رسالة تأكيد لبدء عملية التحقق من البيانات
        session()->flash('info', 'جاري التحقق من بيانات المستخدم...');

        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'email' => 'required|string|email|max:255|unique:users',
            'password' => ['required', 'confirmed', Password::defaults()],
            'branch_id' => 'nullable|exists:branches,id', // تعديل: جعل الفرع اختياريًا
            'phone' => 'nullable|string|max:20',
            'position' => 'nullable|string|max:100',
            'is_active' => 'nullable',
            'roles' => 'nullable|array',
            'roles.*' => 'exists:roles,id',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        // عرض رسالة تأكيد لبدء عملية التحقق من الفرع
        session()->flash('info', 'جاري التحقق من الفرع...');

        // التحقق من الفرع إذا تم تحديده
        $branch = null;
        if ($request->filled('branch_id')) {
            $branch = Branch::findOrFail($request->branch_id);
            // التحقق من أن الفرع ينتمي للشركة (إذا كان المستخدم مرتبطًا بشركة)
            if ($company && $branch->company_id != $company->id) {
                return redirect()->back()
                    ->with('error', 'الفرع المحدد غير صالح.')
                    ->withInput();
            }
        }

        // عرض رسالة تأكيد لبدء عملية إنشاء المستخدم
        session()->flash('info', 'جاري إنشاء المستخدم الجديد...');

        // Create user
        try {
            // تسجيل بيانات الطلب للتشخيص
            \Log::info('User creation request data:', $request->all());

            $newUser = new User();
            $newUser->name = $request->name;
            $newUser->email = $request->email;
            $newUser->password = Hash::make($request->password);

            // تحديد الشركة
            if ($company) {
                $newUser->company_id = $company->id;
            } elseif (isset($branch) && $branch) {
                $newUser->company_id = $branch->company_id;
            } else {
                // إذا لم يتم تحديد شركة أو فرع، يمكن تركها فارغة
                $newUser->company_id = null;
            }

            // تحديد الفرع (يمكن أن يكون فارغًا)
            $newUser->branch_id = $request->branch_id;
            $newUser->phone = $request->phone;
            // تصحيح: استخدام job_title بدلاً من position
            $newUser->job_title = $request->position;

            // تصحيح: تعيين قيمة is_active بشكل صريح
            // استخدام القيمة الافتراضية true إذا لم يتم تحديد القيمة
            if ($request->has('is_active')) {
                // تحويل القيمة إلى boolean بشكل صريح
                $isActive = $request->input('is_active') === 'true' || $request->input('is_active') === '1' || $request->input('is_active') === 1;
                $newUser->is_active = $isActive;
            } else {
                // القيمة الافتراضية هي true
                $newUser->is_active = true;
            }

            // تسجيل قيمة is_active للتحقق
            \Log::info('is_active value from request: ' . ($request->has('is_active') ? $request->input('is_active') : 'not set'));
            \Log::info('is_active value after processing: ' . ($newUser->is_active ? 'true' : 'false'));

            // عرض رسالة تأكيد قبل محاولة الحفظ
            session()->flash('info', 'جاري حفظ بيانات المستخدم...');

            // تسجيل بيانات المستخدم قبل الحفظ
            \Log::info('User data before save:', $newUser->toArray());

            // محاولة حفظ المستخدم مع عرض أي أخطاء قد تحدث
            try {
                $saveResult = $newUser->save();
                \Log::info('Save result: ' . ($saveResult ? 'Success' : 'Failed'));

                if (!$saveResult) {
                    // إذا فشلت عملية الحفظ
                    \Log::error('Failed to save user');
                    return redirect()->back()
                        ->with('error', 'فشل في حفظ المستخدم الجديد. يرجى التحقق من البيانات المدخلة.')
                        ->withInput();
                }
            } catch (\Exception $saveException) {
                // عرض رسالة الخطأ مباشرة على الشاشة
                dd('خطأ في حفظ المستخدم: ' . $saveException->getMessage());
            }

            // تسجيل معرف المستخدم بعد الحفظ
            \Log::info('User saved with ID: ' . $newUser->id);

            // عرض رسالة تأكيد بعد نجاح الحفظ
            session()->flash('info', 'تم حفظ بيانات المستخدم بنجاح. جاري تعيين الأدوار...');
        } catch (\Exception $e) {
            // التقاط أي استثناء قد يحدث أثناء عملية الحفظ
            \Log::error('Exception during user creation: ' . $e->getMessage());
            \Log::error($e->getTraceAsString());

            // عرض رسالة الخطأ مباشرة على الشاشة
            dd('خطأ في إنشاء المستخدم: ' . $e->getMessage());

            return redirect()->back()
                ->with('error', 'حدث خطأ أثناء إنشاء المستخدم: ' . $e->getMessage())
                ->withInput();
        }

        try {
            // عرض رسالة تأكيد لبدء عملية تعيين الأدوار
            session()->flash('info', 'جاري تعيين الأدوار للمستخدم...');

            // تسجيل معلومات الأدوار المطلوبة
            \Log::info('Roles requested:', $request->roles ?? []);

            // Assign roles (if any)
            $roleIds = $request->roles ?? [];
            $roles = empty($roleIds) ? collect([]) : Role::whereIn('id', $roleIds)->get();
            \Log::info('Roles found: ' . $roles->count());

            // Check if user has permission to assign these roles (only if roles were selected)
            if ($roles->isNotEmpty()) {
                foreach ($roles as $role) {
                    \Log::info('Checking role: ' . $role->name);

                    if ($role->name == 'super_admin' && !$user->hasRole('super_admin')) {
                        \Log::warning('User attempted to assign super_admin role without permission');
                        return redirect()->back()
                            ->with('error', 'ليس لديك صلاحية لتعيين دور مدير النظام الأعلى.')
                            ->withInput();
                    }

                    if ($role->name == 'admin' && !$user->hasAnyRole(['super_admin', 'admin'])) {
                        \Log::warning('User attempted to assign admin role without permission');
                        return redirect()->back()
                            ->with('error', 'ليس لديك صلاحية لتعيين دور المدير.')
                            ->withInput();
                    }
                }
            }

            \Log::info('Assigning roles to user ID: ' . $newUser->id);

            // محاولة تعيين الأدوار مع التعامل مع الأخطاء المحتملة
            try {
                if ($roles->isNotEmpty()) {
                    $newUser->assignRole($roles);
                    \Log::info('Roles assigned successfully');
                } else {
                    \Log::info('No roles to assign');
                }
            } catch (\Exception $roleException) {
                // تسجيل الخطأ ولكن عدم إيقاف العملية
                \Log::error('Error assigning roles: ' . $roleException->getMessage());
                \Log::error($roleException->getTraceAsString());
            }

            return redirect()->route('users.show', $newUser->id)
                ->with('success', 'تم إنشاء المستخدم بنجاح.');
        } catch (\Exception $e) {
            // في حالة حدوث خطأ أثناء تعيين الأدوار
            \Log::error('Exception during role assignment: ' . $e->getMessage());
            \Log::error($e->getTraceAsString());

            // حذف المستخدم الذي تم إنشاؤه لتجنب وجود مستخدمين بدون أدوار
            if (isset($newUser) && $newUser->id) {
                \Log::info('Deleting user ID: ' . $newUser->id . ' due to role assignment failure');
                try {
                    $newUser->delete();
                } catch (\Exception $deleteException) {
                    \Log::error('Error deleting user: ' . $deleteException->getMessage());
                }
            }

            return redirect()->back()
                ->with('error', 'حدث خطأ أثناء تعيين الأدوار للمستخدم: ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $currentUser = Auth::user();
        $company = $currentUser->company;

        $user = User::findOrFail($id);

        // تعديل: السماح بعرض المستخدمين حتى لو لم يكن المستخدم الحالي مرتبطًا بشركة
        // إذا كان لديه دور super_admin أو admin
        if (!$company && !$currentUser->isSystemAdmin()) {
            return redirect()->route('companies.index')
                ->with('error', 'يجب أن تكون مرتبطًا بشركة أولاً.');
        }

        // تعديل: التحقق من أن المستخدم ينتمي لنفس الشركة فقط إذا كان المستخدم الحالي مرتبطًا بشركة
        // وليس لديه دور super_admin أو admin
        if ($company && $user->company_id != $company->id && !$currentUser->isSystemAdmin()) {
            return redirect()->route('users.index')
                ->with('error', 'ليس لديك صلاحية لعرض هذا المستخدم.');
        }

        // Get user activity statistics
        // تعديل: استخدام قيم افتراضية بدلاً من العلاقات غير الموجودة
        $statistics = [
            'customers_count' => 0, // سيتم تحديثها لاحقًا عندما يتم إضافة العلاقة
            'elevators_count' => 0, // سيتم تحديثها لاحقًا عندما يتم إضافة العلاقة
            'contracts_count' => 0, // سيتم تحديثها لاحقًا عندما يتم إضافة العلاقة
            'invoices_count' => 0, // سيتم تحديثها لاحقًا عندما يتم إضافة العلاقة
            'last_login' => $user->last_login_at,
        ];

        // إذا كانت العلاقة موجودة، استخدمها
        if (method_exists($user, 'customers') && Schema::hasColumn('customers', 'user_id')) {
            try {
                $statistics['customers_count'] = $user->customers()->count() ?? 0;
            } catch (\Exception $e) {
                \Log::error('Error counting customers: ' . $e->getMessage());
            }
        }

        // إحصائيات الفواتير التي أنشأها المستخدم
        if (method_exists($user, 'createdInvoices')) {
            try {
                $statistics['invoices_count'] = $user->createdInvoices()->count() ?? 0;
            } catch (\Exception $e) {
                \Log::error('Error counting invoices: ' . $e->getMessage());
            }
        }

        return view('users.show', compact('user', 'statistics'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $currentUser = Auth::user();
        $company = $currentUser->company;

        // تعديل: السماح للمستخدم بتعديل المستخدمين حتى لو لم يكن مرتبطًا بشركة
        // إذا كان لديه دور super_admin أو admin
        if (!$company && !$currentUser->isSystemAdmin()) {
            return redirect()->route('companies.index')
                ->with('error', 'You need to be associated with a company first.');
        }

        $user = User::findOrFail($id);

        // Check if user belongs to the same company (إلا إذا كان المستخدم super_admin أو admin)
        if ($company && $user->company_id != $company->id && !$currentUser->isSystemAdmin()) {
            return redirect()->route('users.index')
                ->with('error', 'You do not have permission to edit this user.');
        }

        // Check if current user has permission to edit users
        if (!$currentUser->hasPermissionTo('edit users') && $currentUser->id != $user->id) {
            return redirect()->route('users.index')
                ->with('error', 'You do not have permission to edit other users.');
        }

        // Super admin can only be edited by another super admin
        if ($user->hasRole('super_admin') && !$currentUser->hasRole('super_admin')) {
            return redirect()->route('users.index')
                ->with('error', 'You do not have permission to edit a super admin user.');
        }

        // Get all branches for dropdown
        if ($company) {
            $branches = Branch::where('company_id', $company->id)
                ->where('is_active', true)
                ->orderBy('name')
                ->get();
        } else {
            // إذا كان المستخدم super_admin وليس مرتبطًا بشركة، اجلب جميع الفروع النشطة
            $branches = Branch::where('is_active', true)
                ->orderBy('name')
                ->get();
        }

        // تسجيل معلومات المستخدم الحالي للتشخيص
        \Log::info('Current user: ' . $currentUser->name . ' (ID: ' . $currentUser->id . ')');
        \Log::info('Current user roles: ' . implode(', ', $currentUser->roles()->pluck('name')->toArray()));

        // تعديل: جلب جميع الأدوار بدون قيود
        $roles = Role::orderBy('name')->get();

        // تسجيل الأدوار التي تم جلبها
        \Log::info('Roles fetched for dropdown: ' . $roles->count());
        \Log::info('Roles: ' . implode(', ', $roles->pluck('name')->toArray()));

        return view('users.edit', compact('user', 'branches', 'roles'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $currentUser = Auth::user();
        $company = $currentUser->company;

        // تعديل: السماح للمستخدم بتعديل المستخدمين حتى لو لم يكن مرتبطًا بشركة
        // إذا كان لديه دور super_admin أو admin
        if (!$company && !$currentUser->isSystemAdmin()) {
            return redirect()->route('companies.index')
                ->with('error', 'You need to be associated with a company first.');
        }

        $user = User::findOrFail($id);

        // Check if user belongs to the same company (إلا إذا كان المستخدم super_admin أو admin)
        if ($company && $user->company_id != $company->id && !$currentUser->isSystemAdmin()) {
            return redirect()->route('users.index')
                ->with('error', 'You do not have permission to update this user.');
        }

        // Check if current user has permission to edit users
        if (!$currentUser->hasPermissionTo('edit users') && $currentUser->id != $user->id) {
            return redirect()->route('users.index')
                ->with('error', 'You do not have permission to edit other users.');
        }

        // Super admin can only be edited by another super admin
        if ($user->hasRole('super_admin') && !$currentUser->hasRole('super_admin')) {
            return redirect()->route('users.index')
                ->with('error', 'You do not have permission to edit a super admin user.');
        }

        // Validate input
        $rules = [
            'name' => 'required|string|max:255',
            'email' => 'required|string|email|max:255|unique:users,email,' . $user->id,
            'branch_id' => 'nullable|exists:branches,id', // تعديل: جعل الفرع اختياريًا
            'phone' => 'nullable|string|max:20',
            'position' => 'nullable|string|max:100',
            'is_active' => 'boolean',
        ];

        // تم تعديل الشرط للسماح بتغيير الأدوار لجميع المستخدمين
        $rules['roles'] = 'required|array';
        $rules['roles.*'] = 'exists:roles,id';

        // Password is optional on update
        if ($request->filled('password')) {
            $rules['password'] = ['required', 'confirmed', Password::defaults()];
        }

        $validator = Validator::make($request->all(), $rules);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        // التحقق من الفرع إذا تم تحديده
        if ($request->filled('branch_id')) {
            $branch = Branch::findOrFail($request->branch_id);
            // التحقق من أن الفرع ينتمي للشركة (إذا كان المستخدم مرتبطًا بشركة)
            if ($company && $branch->company_id != $company->id && !$currentUser->hasRole('super_admin')) {
                return redirect()->back()
                    ->with('error', 'Invalid branch selected.')
                    ->withInput();
            }
        }

        // Update user
        $user->name = $request->name;
        $user->email = $request->email;
        if ($request->filled('password')) {
            $user->password = Hash::make($request->password);
        }
        $user->branch_id = $request->branch_id; // يمكن أن يكون null
        $user->phone = $request->phone;
        $user->position = $request->position;

        // Only admin or super_admin can change active status
        if ($currentUser->hasAnyRole(['admin', 'super_admin'])) {
            // تحويل قيمة الحقل إلى boolean
            $user->is_active = $request->has('is_active') ? true : false;
        }

        $user->save();

        // تم تعديل الشرط للسماح بتحديث الأدوار لجميع المستخدمين
        if ($request->has('roles')) {
            $roles = Role::whereIn('id', $request->roles)->get();

            // التحقق من صلاحية تعيين أدوار معينة
            foreach ($roles as $role) {
                // فقط مستخدم super_admin يمكنه تعيين دور super_admin
                if ($role->name == 'super_admin' && !$currentUser->hasRole('super_admin')) {
                    return redirect()->back()
                        ->with('error', 'ليس لديك صلاحية لتعيين دور مدير النظام الأعلى.')
                        ->withInput();
                }

                // فقط مستخدم super_admin أو admin يمكنه تعيين دور admin
                if ($role->name == 'admin' && !$currentUser->hasAnyRole(['super_admin', 'admin'])) {
                    return redirect()->back()
                        ->with('error', 'ليس لديك صلاحية لتعيين دور المدير.')
                        ->withInput();
                }
            }

            // مزامنة الأدوار
            $user->roles()->sync($roles);

            // تسجيل الأدوار التي تم تعيينها
            \Log::info('تم تحديث أدوار المستخدم ' . $user->name . ' (ID: ' . $user->id . '): ' . implode(', ', $roles->pluck('name')->toArray()));
        }

        return redirect()->route('users.show', $user->id)
            ->with('success', 'User updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        $currentUser = Auth::user();
        $company = $currentUser->company;

        // تعديل: السماح بحذف المستخدمين حتى لو لم يكن المستخدم الحالي مرتبطًا بشركة
        // إذا كان لديه دور super_admin أو admin
        if (!$company && !$currentUser->isSystemAdmin()) {
            return redirect()->route('companies.index')
                ->with('error', 'You need to be associated with a company first.');
        }

        // Cannot delete yourself
        if ($currentUser->id == $id) {
            return redirect()->route('users.index')
                ->with('error', 'You cannot delete your own account.');
        }

        // Check if current user has permission to delete users
        if (!$currentUser->hasPermissionTo('delete users')) {
            return redirect()->route('users.index')
                ->with('error', 'You do not have permission to delete users.');
        }

        $user = User::findOrFail($id);

        // Check if user belongs to the same company (إلا إذا كان المستخدم super_admin أو admin)
        if ($company && $user->company_id != $company->id && !$currentUser->isSystemAdmin()) {
            return redirect()->route('users.index')
                ->with('error', 'You do not have permission to delete this user.');
        }

        // Super admin can only be deleted by another super admin
        if ($user->hasRole('super_admin') && !$currentUser->hasRole('super_admin')) {
            return redirect()->route('users.index')
                ->with('error', 'You do not have permission to delete a super admin user.');
        }

        // Check if user has related records
        $hasRelatedRecords =
            $user->customers()->count() > 0 ||
            $user->elevators()->count() > 0 ||
            $user->maintenance_contracts()->count() > 0 ||
            $user->invoices()->count() > 0;

        if ($hasRelatedRecords) {
            return redirect()->route('users.index')
                ->with('error', 'This user has related records and cannot be deleted. You can deactivate the account instead.');
        }

        // Delete user
        $user->delete();

        return redirect()->route('users.index')
            ->with('success', 'User deleted successfully.');
    }
}
