<?php

namespace App\Http\Controllers;

use App\Models\User;
use App\Models\Role;
use App\Models\Branch;
use App\Models\Company;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;

class UserArabicController extends Controller
{
    /**
     * عرض قائمة المستخدمين
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        $companyId = session('company_id');
        $query = User::where('company_id', $companyId);

        // تطبيق الفلتر حسب الفرع
        if ($request->filled('branch_id')) {
            $query->where('branch_id', $request->branch_id);
        }

        // تطبيق الفلتر حسب الدور
        if ($request->filled('role')) {
            $roleName = $request->role;
            $query->whereHas('roles', function ($q) use ($roleName) {
                $q->where('name', $roleName);
            });
        }

        // ترتيب المستخدمين حسب الاسم
        $query->orderBy('name', 'asc');

        // تقسيم النتائج إلى صفحات
        $users = $query->paginate(10);

        $roles = Role::all();
        $branches = Branch::where('company_id', $companyId)->get();

        return view('users.index-arabic', compact('users', 'roles', 'branches'));
    }

    /**
     * عرض نموذج إنشاء مستخدم جديد
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $companyId = session('company_id');
        $roles = Role::all();
        $branches = Branch::where('company_id', $companyId)->get();
        $companies = Company::all();

        return view('users.create-arabic', compact('roles', 'branches', 'companies'));
    }

    /**
     * تخزين مستخدم جديد في قاعدة البيانات
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'email' => 'required|string|email|max:255|unique:users',
            'password' => 'required|string|min:8|confirmed',
            'company_id' => 'required|exists:companies,id',
            'branch_id' => 'nullable|exists:branches,id',
            'roles' => 'required|array',
            'roles.*' => 'exists:roles,id',
        ], [
            'name.required' => 'حقل الاسم مطلوب',
            'email.required' => 'حقل البريد الإلكتروني مطلوب',
            'email.email' => 'يجب أن يكون البريد الإلكتروني صالحًا',
            'email.unique' => 'البريد الإلكتروني مستخدم بالفعل',
            'password.required' => 'حقل كلمة المرور مطلوب',
            'password.min' => 'يجب أن تتكون كلمة المرور من 8 أحرف على الأقل',
            'password.confirmed' => 'تأكيد كلمة المرور غير متطابق',
            'company_id.required' => 'حقل الشركة مطلوب',
            'roles.required' => 'يجب اختيار دور واحد على الأقل',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $user = User::create([
            'name' => $request->name,
            'email' => $request->email,
            'password' => Hash::make($request->password),
            'company_id' => $request->company_id,
            'branch_id' => $request->branch_id,
            'is_active' => $request->has('is_active') ? 1 : 0,
        ]);

        $user->roles()->attach($request->roles);

        return redirect()->route('users.arabic.index')
            ->with('success', 'تم إنشاء المستخدم بنجاح');
    }

    /**
     * عرض معلومات مستخدم محدد
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $companyId = session('company_id');
        $user = User::where('company_id', $companyId)->findOrFail($id);

        return view('users.show-arabic', compact('user'));
    }

    /**
     * عرض نموذج تعديل مستخدم
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $companyId = session('company_id');
        $user = User::where('company_id', $companyId)->findOrFail($id);
        $roles = Role::all();
        $branches = Branch::where('company_id', $companyId)->get();
        $companies = Company::all();

        return view('users.edit-arabic', compact('user', 'roles', 'branches', 'companies'));
    }

    /**
     * تحديث معلومات مستخدم محدد
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $companyId = session('company_id');
        $user = User::where('company_id', $companyId)->findOrFail($id);

        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'email' => 'required|string|email|max:255|unique:users,email,' . $id,
            'password' => 'nullable|string|min:8|confirmed',
            'company_id' => 'required|exists:companies,id',
            'branch_id' => 'nullable|exists:branches,id',
            'roles' => 'required|array',
            'roles.*' => 'exists:roles,id',
        ], [
            'name.required' => 'حقل الاسم مطلوب',
            'email.required' => 'حقل البريد الإلكتروني مطلوب',
            'email.email' => 'يجب أن يكون البريد الإلكتروني صالحًا',
            'email.unique' => 'البريد الإلكتروني مستخدم بالفعل',
            'password.min' => 'يجب أن تتكون كلمة المرور من 8 أحرف على الأقل',
            'password.confirmed' => 'تأكيد كلمة المرور غير متطابق',
            'company_id.required' => 'حقل الشركة مطلوب',
            'roles.required' => 'يجب اختيار دور واحد على الأقل',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $userData = [
            'name' => $request->name,
            'email' => $request->email,
            'company_id' => $request->company_id,
            'branch_id' => $request->branch_id,
            'is_active' => $request->has('is_active') ? 1 : 0,
        ];

        if ($request->filled('password')) {
            $userData['password'] = Hash::make($request->password);
        }

        $user->update($userData);

        // تحديث الأدوار
        $user->roles()->sync($request->roles);

        return redirect()->route('users.arabic.index')
            ->with('success', 'تم تحديث المستخدم بنجاح');
    }

    /**
     * حذف مستخدم محدد
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        $companyId = session('company_id');
        $user = User::where('company_id', $companyId)->findOrFail($id);

        // التحقق من أن المستخدم لا يحذف نفسه
        if ($user->id === Auth::id()) {
            return redirect()->route('users.arabic.index')
                ->with('error', 'لا يمكنك حذف حسابك الحالي');
        }

        // حذف علاقات الأدوار أولاً
        $user->roles()->detach();

        // حذف المستخدم
        $user->delete();

        return redirect()->route('users.arabic.index')
            ->with('success', 'تم حذف المستخدم بنجاح');
    }

    /**
     * تغيير حالة المستخدم (نشط/غير نشط)
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function toggleStatus($id)
    {
        $companyId = session('company_id');
        $user = User::where('company_id', $companyId)->findOrFail($id);

        // التحقق من أن المستخدم لا يغير حالة نفسه
        if ($user->id === Auth::id()) {
            return redirect()->route('users.arabic.index')
                ->with('error', 'لا يمكنك تغيير حالة حسابك الحالي');
        }

        $user->is_active = !$user->is_active;
        $user->save();

        $status = $user->is_active ? 'نشط' : 'غير نشط';

        return redirect()->route('users.arabic.index')
            ->with('success', "تم تغيير حالة المستخدم إلى $status بنجاح");
    }
}
