<?php

namespace App\Http\Controllers;

use App\Models\TwoFactorToken;
use App\Services\TwoFactorAuthService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\ValidationException;

class TwoFactorAuthController extends Controller
{
    /**
     * خدمة المصادقة الثنائية.
     *
     * @var \App\Services\TwoFactorAuthService
     */
    protected $twoFactorService;

    /**
     * إنشاء مثيل جديد من وحدة التحكم.
     *
     * @param \App\Services\TwoFactorAuthService $twoFactorService
     * @return void
     */
    public function __construct(TwoFactorAuthService $twoFactorService)
    {
        $this->twoFactorService = $twoFactorService;
        $this->middleware('auth');
    }

    /**
     * عرض صفحة إعدادات المصادقة الثنائية.
     *
     * @return \Illuminate\View\View
     */
    public function index()
    {
        $user = Auth::user();

        return view('auth.two-factor.index', [
            'user' => $user,
            'enabled' => $user->hasTwoFactorEnabled(),
        ]);
    }

    /**
     * عرض صفحة تفعيل المصادقة الثنائية.
     *
     * @return \Illuminate\View\View
     */
    public function setup()
    {
        $user = Auth::user();

        // التحقق من كلمة المرور قبل إعداد المصادقة الثنائية
        if (!session('auth_confirmed')) {
            return redirect()->route('two-factor.confirm-password');
        }

        // إنشاء مفتاح سري جديد إذا لم يكن موجوداً
        if (!$user->two_factor_secret) {
            $secretKey = $this->twoFactorService->generateSecretKey($user);
        } else {
            $secretKey = decrypt($user->two_factor_secret);
        }

        // إنشاء رابط QR
        $qrCodeUrl = $this->twoFactorService->getQrCodeUrl($user, $secretKey);

        // إنشاء رموز الاسترداد إذا لم تكن موجودة
        if (!$user->two_factor_recovery_codes) {
            $recoveryCodes = $this->twoFactorService->generateRecoveryCodes($user);
        } else {
            $recoveryCodes = json_decode(decrypt($user->two_factor_recovery_codes), true);
        }

        return view('auth.two-factor.setup', [
            'secretKey' => $secretKey,
            'qrCodeUrl' => $qrCodeUrl,
            'recoveryCodes' => $recoveryCodes,
        ]);
    }

    /**
     * تأكيد كلمة المرور قبل إعداد المصادقة الثنائية.
     *
     * @return \Illuminate\View\View
     */
    public function confirmPassword()
    {
        return view('auth.two-factor.confirm-password');
    }

    /**
     * التحقق من كلمة المرور.
     *
     * @param \Illuminate\Http\Request $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function validatePassword(Request $request)
    {
        $request->validate([
            'password' => 'required|string',
        ]);

        $user = Auth::user();

        if (!Hash::check($request->password, $user->password)) {
            throw ValidationException::withMessages([
                'password' => [__('كلمة المرور غير صحيحة.')],
            ]);
        }

        $request->session()->put('auth_confirmed', true);
        $request->session()->put('auth_confirmed_at', time());

        return redirect()->route('two-factor.setup');
    }

    /**
     * تفعيل المصادقة الثنائية.
     *
     * @param \Illuminate\Http\Request $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function enable(Request $request)
    {
        $request->validate([
            'code' => 'required|string|size:6',
        ]);

        $user = Auth::user();

        if ($this->twoFactorService->enable($user, $request->code)) {
            return redirect()->route('two-factor.index')
                ->with('success', 'تم تفعيل المصادقة الثنائية بنجاح.');
        }

        return back()->withErrors([
            'code' => 'الرمز غير صحيح. يرجى المحاولة مرة أخرى.',
        ]);
    }

    /**
     * تعطيل المصادقة الثنائية.
     *
     * @param \Illuminate\Http\Request $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function disable(Request $request)
    {
        $request->validate([
            'password' => 'required|string',
        ]);

        $user = Auth::user();

        if (!Hash::check($request->password, $user->password)) {
            throw ValidationException::withMessages([
                'password' => [__('كلمة المرور غير صحيحة.')],
            ]);
        }

        $this->twoFactorService->disable($user);

        return redirect()->route('two-factor.index')
            ->with('success', 'تم تعطيل المصادقة الثنائية بنجاح.');
    }

    /**
     * إعادة إنشاء رموز الاسترداد.
     *
     * @param \Illuminate\Http\Request $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function regenerateRecoveryCodes(Request $request)
    {
        $user = Auth::user();

        // التحقق من كلمة المرور قبل إعادة إنشاء رموز الاسترداد
        if (!session('auth_confirmed') || (time() - session('auth_confirmed_at')) > 300) {
            return redirect()->route('two-factor.confirm-password');
        }

        $recoveryCodes = $this->twoFactorService->generateRecoveryCodes($user);

        return redirect()->route('two-factor.setup')
            ->with('success', 'تم إعادة إنشاء رموز الاسترداد بنجاح.');
    }

    /**
     * عرض صفحة التحقق من المصادقة الثنائية.
     *
     * @return \Illuminate\View\View
     */
    public function verify()
    {
        return view('auth.two-factor.verify');
    }

    /**
     * التحقق من رمز المصادقة الثنائية.
     *
     * @param \Illuminate\Http\Request $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function validateCode(Request $request)
    {
        $request->validate([
            'code' => 'required|string',
        ]);

        $user = Auth::user();

        // التحقق من رمز المصادقة
        if (strlen($request->code) === 6 && $this->twoFactorService->verifyCode($user, $request->code)) {
            $request->session()->put('two_factor_authenticated', true);
            return redirect()->intended(route('dashboard'));
        }

        // التحقق من رمز الاسترداد
        if (strlen($request->code) === 16 && $this->twoFactorService->verifyRecoveryCode($user, $request->code)) {
            $request->session()->put('two_factor_authenticated', true);
            return redirect()->intended(route('dashboard'))
                ->with('warning', 'تم استخدام رمز استرداد. يجب إنشاء رموز استرداد جديدة.');
        }

        return back()->withErrors([
            'code' => 'الرمز غير صحيح. يرجى المحاولة مرة أخرى.',
        ]);
    }

    /**
     * إرسال رمز مصادقة ثنائية عبر البريد الإلكتروني.
     *
     * @param \Illuminate\Http\Request $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function sendEmailToken(Request $request)
    {
        $user = Auth::user();

        // إنشاء رمز مؤقت
        $token = $this->twoFactorService->generateToken($user);

        // إرسال الرمز عبر البريد الإلكتروني
        // Mail::to($user->email)->send(new TwoFactorCode($token));

        return back()->with('success', 'تم إرسال رمز المصادقة إلى بريدك الإلكتروني.');
    }
}
