<?php

namespace App\Http\Controllers;

use App\Models\Permission;
use App\Models\Role;
use App\Models\User;
use App\Models\UserPermission;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Illuminate\Validation\Rule;

class TempPermissionController extends Controller
{
    /**
     * إنشاء مثيل جديد من وحدة التحكم.
     *
     * @return void
     */
    public function __construct()
    {
        $this->middleware('auth');
        // لا نستخدم وسيط الصلاحيات هنا
    }

    /**
     * عرض قائمة المستخدمين لإدارة الصلاحيات المباشرة.
     *
     * @return \Illuminate\View\View
     */
    public function manageUsers()
    {
        $currentUser = auth()->user();
        $company = $currentUser->company;

        // إذا كان المستخدم مرتبطاً بشركة، نعرض فقط مستخدمي تلك الشركة
        // وإلا نعرض جميع المستخدمين (للمستخدم التجريبي أو المدير العام)
        if ($company && !$currentUser->hasRole('super_admin')) {
            $users = User::where('company_id', $company->id)
                ->orderBy('name')
                ->paginate(15);
        } else {
            $users = User::orderBy('name')->paginate(15);
        }

        return view('admin.permissions.manage-users', compact('users'));
    }

    /**
     * عرض صفحة إدارة صلاحيات المستخدم المباشرة.
     *
     * @param \App\Models\User $user
     * @return \Illuminate\View\View
     */
    public function editUserPermissions(User $user)
    {
        $currentUser = auth()->user();
        $company = $currentUser->company;

        // التحقق من أن المستخدم ينتمي إلى نفس الشركة أو أن المستخدم الحالي غير مرتبط بأي شركة
        if ($currentUser->company_id !== null && $user->company_id != $currentUser->company_id) {
            abort(403, 'ليس لديك صلاحية للوصول إلى هذا المستخدم');
        }

        $permissions = Permission::orderBy('group')
            ->orderBy('name')
            ->get()
            ->groupBy('group');

        $userPermissions = $user->directPermissions()
            ->with('permission')
            ->get()
            ->keyBy('permission_id');

        $rolePermissions = $user->getAllPermissions()
            ->pluck('id')
            ->toArray();

        return view('admin.permissions.edit-user-permissions', compact(
            'user',
            'permissions',
            'userPermissions',
            'rolePermissions'
        ));
    }

    /**
     * تحديث صلاحيات المستخدم المباشرة.
     *
     * @param \Illuminate\Http\Request $request
     * @param \App\Models\User $user
     * @return \Illuminate\Http\RedirectResponse
     */
    public function updateUserPermissions(Request $request, User $user)
    {
        $currentUser = auth()->user();
        $company = $currentUser->company;

        // التحقق من أن المستخدم ينتمي إلى نفس الشركة أو أن المستخدم الحالي هو super_admin
        if ($company && !$currentUser->hasRole('super_admin') && $user->company_id != $company->id) {
            abort(403, 'ليس لديك صلاحية للوصول إلى هذا المستخدم');
        }

        $request->validate([
            'permissions' => 'nullable|array',
            'permissions.*.id' => 'required|exists:permissions,id',
            'permissions.*.granted' => 'nullable',
            'permissions.*.expires_at' => 'nullable|date',
        ]);

        DB::beginTransaction();

        try {
            // حذف جميع الصلاحيات المباشرة الحالية
            $user->directPermissions()->delete();

            // إضافة الصلاحيات المباشرة الجديدة
            if ($request->has('permissions')) {
                foreach ($request->permissions as $permissionData) {
                    if (isset($permissionData['id']) && isset($permissionData['granted']) && $permissionData['granted'] !== '') {
                        $user->directPermissions()->create([
                            'permission_id' => $permissionData['id'],
                            'granted' => $permissionData['granted'] ? true : false,
                            'source' => 'admin',
                            'expires_at' => $permissionData['expires_at'] ?? null,
                        ]);
                    }
                }
            }

            DB::commit();

            return redirect()->route('temp.permissions.manage-users')
                ->with('success', 'تم تحديث صلاحيات المستخدم بنجاح.');
        } catch (\Exception $e) {
            DB::rollBack();

            return back()->withInput()->withErrors([
                'error' => 'حدث خطأ أثناء تحديث صلاحيات المستخدم: ' . $e->getMessage(),
            ]);
        }
    }
}
