<?php

namespace App\Http\Controllers;

use App\Models\TaxSetting;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class TaxSettingController extends Controller
{
    /**
     * إنشاء مثيل جديد من الكنترولر.
     *
     * @return void
     */
    public function __construct()
    {
        $this->middleware('permission:tax_settings.view')->only(['index']);
        $this->middleware('permission:tax_settings.create')->only(['create', 'store']);
        $this->middleware('permission:tax_settings.edit')->only(['edit', 'update', 'setDefault']);
        $this->middleware('permission:tax_settings.delete')->only(['destroy']);
    }
    /**
     * عرض قائمة إعدادات الضرائب.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $companyId = auth()->user()->company_id;
        $taxSettings = TaxSetting::where('company_id', $companyId)
            ->orderBy('is_default', 'desc')
            ->orderBy('name')
            ->get();

        return view('tax_settings.index', compact('taxSettings'));
    }

    /**
     * عرض نموذج إنشاء إعداد ضريبة جديد.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        return view('tax_settings.create');
    }

    /**
     * تخزين إعداد ضريبة جديد.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'rate' => 'required|numeric|min:0|max:100',
            'is_default' => 'boolean',
            'is_active' => 'boolean',
            'description' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $companyId = auth()->user()->company_id;

        // إذا كان الإعداد الجديد هو الافتراضي، يجب إلغاء تعيين الإعداد الافتراضي الحالي
        if ($request->is_default) {
            TaxSetting::where('company_id', $companyId)
                ->where('is_default', true)
                ->update(['is_default' => false]);
        }

        TaxSetting::create([
            'company_id' => $companyId,
            'name' => $request->name,
            'rate' => $request->rate,
            'is_default' => $request->is_default ?? false,
            'is_active' => $request->is_active ?? true,
            'description' => $request->description,
        ]);

        return redirect()->route('tax_settings.index')
            ->with('success', 'تم إنشاء إعداد الضريبة بنجاح');
    }

    /**
     * عرض نموذج تعديل إعداد ضريبة.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $companyId = auth()->user()->company_id;
        $taxSetting = TaxSetting::where('company_id', $companyId)->findOrFail($id);

        return view('tax_settings.edit', compact('taxSetting'));
    }

    /**
     * تحديث إعداد ضريبة محدد.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'rate' => 'required|numeric|min:0|max:100',
            'is_default' => 'boolean',
            'is_active' => 'boolean',
            'description' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $companyId = auth()->user()->company_id;
        $taxSetting = TaxSetting::where('company_id', $companyId)->findOrFail($id);

        // إذا كان الإعداد الجديد هو الافتراضي، يجب إلغاء تعيين الإعداد الافتراضي الحالي
        if ($request->is_default && !$taxSetting->is_default) {
            TaxSetting::where('company_id', $companyId)
                ->where('is_default', true)
                ->update(['is_default' => false]);
        }

        $taxSetting->update([
            'name' => $request->name,
            'rate' => $request->rate,
            'is_default' => $request->is_default ?? false,
            'is_active' => $request->is_active ?? true,
            'description' => $request->description,
        ]);

        return redirect()->route('tax_settings.index')
            ->with('success', 'تم تحديث إعداد الضريبة بنجاح');
    }

    /**
     * حذف إعداد ضريبة محدد.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        $companyId = auth()->user()->company_id;
        $taxSetting = TaxSetting::where('company_id', $companyId)->findOrFail($id);

        // لا يمكن حذف الإعداد الافتراضي
        if ($taxSetting->is_default) {
            return redirect()->back()
                ->with('error', 'لا يمكن حذف إعداد الضريبة الافتراضي');
        }

        try {
            $taxSetting->delete();

            return redirect()->route('tax_settings.index')
                ->with('success', 'تم حذف إعداد الضريبة بنجاح');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'حدث خطأ أثناء حذف إعداد الضريبة: ' . $e->getMessage());
        }
    }

    /**
     * تعيين إعداد ضريبة كافتراضي.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function setDefault($id)
    {
        $companyId = auth()->user()->company_id;
        $taxSetting = TaxSetting::where('company_id', $companyId)->findOrFail($id);

        // إلغاء تعيين الإعداد الافتراضي الحالي
        TaxSetting::where('company_id', $companyId)
            ->where('is_default', true)
            ->update(['is_default' => false]);

        // تعيين الإعداد الجديد كافتراضي
        $taxSetting->update(['is_default' => true]);

        return redirect()->route('tax_settings.index')
            ->with('success', 'تم تعيين إعداد الضريبة كافتراضي بنجاح');
    }
}
