<?php

namespace App\Http\Controllers;

use App\Models\TaxReport;
use App\Models\Company;
use App\Services\InvoiceService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class TaxReportController extends Controller
{
    protected $invoiceService;

    /**
     * Create a new controller instance.
     *
     * @param  InvoiceService  $invoiceService
     * @return void
     */
    public function __construct(InvoiceService $invoiceService)
    {
        $this->invoiceService = $invoiceService;
        $this->middleware('auth');

        // تعديل الصلاحية للسماح بالوصول للمدير العام والمدير والمحاسب
        $this->middleware(function ($request, $next) {
            if (
                auth()->user()->hasRole('super_admin') ||
                auth()->user()->hasRole('admin') ||
                auth()->user()->hasRole('accountant')
            ) {
                return $next($request);
            }

            if (auth()->user()->hasPermissionTo('tax_reports.view')) {
                return $next($request);
            }

            abort(403, 'ليس لديك صلاحية للوصول إلى هذه الصفحة');
        });
    }

    /**
     * عرض قائمة تقارير الضرائب.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $companyId = auth()->user()->company_id;
        $taxReports = TaxReport::where('company_id', $companyId)
            ->orderBy('created_at', 'desc')
            ->paginate(15);

        return view('tax_reports.index', compact('taxReports'));
    }

    /**
     * عرض نموذج إنشاء تقرير ضريبة جديد.
     *
     * @return \Illuminate\Http\Response
     */
    public function generateForm()
    {
        return view('tax_reports.generate');
    }

    /**
     * إنشاء تقرير ضريبة جديد.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function generate(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'start_date' => 'required|date',
            'end_date' => 'required|date|after_or_equal:start_date',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $companyId = auth()->user()->company_id;
        $startDate = $request->input('start_date');
        $endDate = $request->input('end_date');

        $taxReport = $this->invoiceService->generateTaxReport(
            $companyId,
            $startDate,
            $endDate
        );

        return redirect()->route('tax-reports.show', $taxReport->id)
            ->with('success', 'تم إنشاء تقرير الضريبة بنجاح.');
    }

    /**
     * عرض تقرير ضريبة محدد.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $companyId = auth()->user()->company_id;
        $taxReport = TaxReport::where('company_id', $companyId)
            ->with('company')
            ->findOrFail($id);

        // الحصول على الفواتير المشمولة في التقرير
        $invoices = $taxReport->invoices();

        return view('tax_reports.show', compact('taxReport', 'invoices'));
    }

    /**
     * عرض نموذج تعديل تقرير ضريبة.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $companyId = auth()->user()->company_id;
        $taxReport = TaxReport::where('company_id', $companyId)
            ->findOrFail($id);

        return view('tax_reports.edit', compact('taxReport'));
    }

    /**
     * تحديث تقرير ضريبة محدد.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $validator = Validator::make($request->all(), [
            'status' => 'required|in:draft,submitted,approved,rejected',
            'notes' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $companyId = auth()->user()->company_id;
        $taxReport = TaxReport::where('company_id', $companyId)
            ->findOrFail($id);

        $taxReport->update([
            'status' => $request->input('status'),
            'notes' => $request->input('notes'),
        ]);

        return redirect()->route('tax-reports.index')
            ->with('success', 'تم تحديث تقرير الضريبة بنجاح.');
    }

    /**
     * حذف تقرير ضريبة محدد.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        $companyId = auth()->user()->company_id;
        $taxReport = TaxReport::where('company_id', $companyId)
            ->findOrFail($id);

        // التحقق من أن التقرير ليس في حالة مقدم أو معتمد
        if (in_array($taxReport->status, ['submitted', 'approved'])) {
            return redirect()->route('tax-reports.index')
                ->with('error', 'لا يمكن حذف تقرير ضريبة مقدم أو معتمد.');
        }

        $taxReport->delete();

        return redirect()->route('tax-reports.index')
            ->with('success', 'تم حذف تقرير الضريبة بنجاح.');
    }

    /**
     * تصدير تقرير ضريبة.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function export($id)
    {
        $companyId = auth()->user()->company_id;
        $taxReport = TaxReport::where('company_id', $companyId)
            ->with('company')
            ->findOrFail($id);

        // الحصول على الفواتير المشمولة في التقرير
        $invoices = $taxReport->invoices();

        // إنشاء محتوى التقرير
        $content = $this->generateTaxReportContent($taxReport, $invoices);

        // تحديد نوع المحتوى واسم الملف
        $filename = 'tax_report_' . $taxReport->report_number . '.pdf';

        // في هذه المرحلة سنعيد المحتوى كنص HTML
        return response($content)
            ->header('Content-Type', 'text/html');

        // يمكن استخدام مكتبة لتحويل HTML إلى PDF
        // return response($pdf)
        //     ->header('Content-Type', 'application/pdf')
        //     ->header('Content-Disposition', 'attachment; filename="' . $filename . '"');
    }

    /**
     * إنشاء محتوى تقرير الضريبة.
     *
     * @param  TaxReport  $taxReport
     * @param  Collection  $invoices
     * @return string
     */
    protected function generateTaxReportContent($taxReport, $invoices)
    {
        $company = $taxReport->company;

        $invoicesHtml = '';
        foreach ($invoices as $invoice) {
            $invoicesHtml .= '<tr>
                <td>' . $invoice->invoice_number . '</td>
                <td>' . $invoice->invoice_date->format('Y-m-d') . '</td>
                <td>' . $invoice->customer->name . '</td>
                <td>' . number_format($invoice->subtotal, 2) . '</td>
                <td>' . number_format($invoice->tax_amount, 2) . '</td>
                <td>' . number_format($invoice->total_amount, 2) . '</td>
            </tr>';
        }

        $html = '<!DOCTYPE html>
        <html lang="ar" dir="rtl">
        <head>
            <meta charset="UTF-8">
            <title>تقرير ضريبة القيمة المضافة</title>
            <style>
                body {
                    font-family: Arial, sans-serif;
                    margin: 20px;
                }
                .report-header {
                    text-align: center;
                    margin-bottom: 30px;
                }
                .company-info {
                    margin-bottom: 20px;
                }
                .report-details {
                    margin-bottom: 30px;
                }
                .report-summary {
                    margin-bottom: 30px;
                }
                table {
                    width: 100%;
                    border-collapse: collapse;
                }
                th, td {
                    border: 1px solid #ddd;
                    padding: 8px;
                    text-align: right;
                }
                th {
                    background-color: #f2f2f2;
                }
                .total-row {
                    font-weight: bold;
                }
            </style>
        </head>
        <body>
            <div class="report-header">
                <h1>تقرير ضريبة القيمة المضافة</h1>
                <p>رقم التقرير: ' . $taxReport->report_number . '</p>
            </div>
            
            <div class="company-info">
                <h2>معلومات الشركة</h2>
                <p><strong>اسم الشركة:</strong> ' . $company->name . '</p>
                <p><strong>العنوان:</strong> ' . $company->address . '</p>
                <p><strong>رقم الهاتف:</strong> ' . $company->phone . '</p>
                <p><strong>البريد الإلكتروني:</strong> ' . $company->email . '</p>
            </div>
            
            <div class="report-details">
                <h2>تفاصيل التقرير</h2>
                <p><strong>الفترة:</strong> من ' . $taxReport->start_date->format('Y-m-d') . ' إلى ' . $taxReport->end_date->format('Y-m-d') . '</p>
                <p><strong>الحالة:</strong> ' . $taxReport->status . '</p>
                <p><strong>تاريخ الإنشاء:</strong> ' . $taxReport->created_at->format('Y-m-d') . '</p>
            </div>
            
            <div class="report-summary">
                <h2>ملخص التقرير</h2>
                <table>
                    <tr>
                        <th>البند</th>
                        <th>المبلغ</th>
                    </tr>
                    <tr>
                        <td>إجمالي ضريبة القيمة المضافة المحصلة</td>
                        <td>' . number_format($taxReport->total_tax_collected, 2) . '</td>
                    </tr>
                    <tr>
                        <td>إجمالي ضريبة القيمة المضافة المدفوعة</td>
                        <td>' . number_format($taxReport->total_tax_paid, 2) . '</td>
                    </tr>
                    <tr class="total-row">
                        <td>صافي ضريبة القيمة المضافة</td>
                        <td>' . number_format($taxReport->net_tax, 2) . '</td>
                    </tr>
                </table>
            </div>
            
            <div class="invoices-list">
                <h2>قائمة الفواتير</h2>
                <table>
                    <thead>
                        <tr>
                            <th>رقم الفاتورة</th>
                            <th>تاريخ الفاتورة</th>
                            <th>العميل</th>
                            <th>المبلغ قبل الضريبة</th>
                            <th>مبلغ الضريبة</th>
                            <th>إجمالي الفاتورة</th>
                        </tr>
                    </thead>
                    <tbody>
                        ' . $invoicesHtml . '
                    </tbody>
                    <tfoot>
                        <tr class="total-row">
                            <td colspan="4">الإجمالي</td>
                            <td>' . number_format($taxReport->total_tax_collected, 2) . '</td>
                            <td>' . number_format($invoices->sum('total_amount'), 2) . '</td>
                        </tr>
                    </tfoot>
                </table>
            </div>
            
            <div class="notes">
                <h2>ملاحظات</h2>
                <p>' . $taxReport->notes . '</p>
            </div>
        </body>
        </html>';

        return $html;
    }
}
