<?php

namespace App\Http\Controllers;

use App\Models\Company;
use App\Models\TaxBracket;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class TaxBracketController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        $companyId = $request->input('company_id', Auth::user()->company_id);

        $taxBrackets = TaxBracket::where('company_id', $companyId)
            ->orderBy('min_amount')
            ->paginate(10);

        return view('hrm.tax_brackets.index', compact('taxBrackets'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $companies = Company::all();

        return view('hrm.tax_brackets.create', compact('companies'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $request->validate([
            'company_id' => 'required|exists:companies,id',
            'name' => 'required|string|max:255',
            'min_amount' => 'required|numeric|min:0',
            'max_amount' => 'required|numeric|min:0|gte:min_amount',
            'tax_rate' => 'required|numeric|min:0|max:100',
            'is_active' => 'boolean',
        ]);

        try {
            $taxBracket = new TaxBracket();
            $taxBracket->company_id = $request->company_id;
            $taxBracket->name = $request->name;
            $taxBracket->min_amount = $request->min_amount;
            $taxBracket->max_amount = $request->max_amount;
            $taxBracket->tax_rate = $request->tax_rate;
            $taxBracket->is_active = $request->has('is_active');
            $taxBracket->created_by = Auth::id();
            $taxBracket->save();

            return redirect()->route('hrm.tax-brackets.index')
                ->with('success', 'تم إنشاء شريحة الضريبة بنجاح');
        } catch (\Exception $e) {
            return back()->with('error', 'حدث خطأ أثناء إنشاء شريحة الضريبة: ' . $e->getMessage());
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $taxBracket = TaxBracket::findOrFail($id);

        // Check if user has access to this tax bracket
        if (Auth::user()->company_id != $taxBracket->company_id && !Auth::user()->isSystemAdmin()) {
            return redirect()->route('hrm.tax-brackets.index')
                ->with('error', 'ليس لديك صلاحية للوصول إلى شريحة الضريبة هذه');
        }

        return view('hrm.tax_brackets.show', compact('taxBracket'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $taxBracket = TaxBracket::findOrFail($id);

        // Check if user has access to this tax bracket
        if (Auth::user()->company_id != $taxBracket->company_id && !Auth::user()->isSystemAdmin()) {
            return redirect()->route('hrm.tax-brackets.index')
                ->with('error', 'ليس لديك صلاحية لتعديل شريحة الضريبة هذه');
        }

        $companies = Company::all();

        return view('hrm.tax_brackets.edit', compact('taxBracket', 'companies'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'min_amount' => 'required|numeric|min:0',
            'max_amount' => 'required|numeric|min:0|gte:min_amount',
            'tax_rate' => 'required|numeric|min:0|max:100',
            'is_active' => 'boolean',
        ]);

        $taxBracket = TaxBracket::findOrFail($id);

        // Check if user has access to this tax bracket
        if (Auth::user()->company_id != $taxBracket->company_id && !Auth::user()->isSystemAdmin()) {
            return redirect()->route('hrm.tax-brackets.index')
                ->with('error', 'ليس لديك صلاحية لتعديل شريحة الضريبة هذه');
        }

        try {
            $taxBracket->name = $request->name;
            $taxBracket->min_amount = $request->min_amount;
            $taxBracket->max_amount = $request->max_amount;
            $taxBracket->tax_rate = $request->tax_rate;
            $taxBracket->is_active = $request->has('is_active');
            $taxBracket->updated_by = Auth::id();
            $taxBracket->save();

            return redirect()->route('hrm.tax-brackets.show', $taxBracket->id)
                ->with('success', 'تم تحديث شريحة الضريبة بنجاح');
        } catch (\Exception $e) {
            return back()->with('error', 'حدث خطأ أثناء تحديث شريحة الضريبة: ' . $e->getMessage());
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        $taxBracket = TaxBracket::findOrFail($id);

        // Check if user has access to this tax bracket
        if (Auth::user()->company_id != $taxBracket->company_id && !Auth::user()->isSystemAdmin()) {
            return redirect()->route('hrm.tax-brackets.index')
                ->with('error', 'ليس لديك صلاحية لحذف شريحة الضريبة هذه');
        }

        try {
            $taxBracket->delete();

            return redirect()->route('hrm.tax-brackets.index')
                ->with('success', 'تم حذف شريحة الضريبة بنجاح');
        } catch (\Exception $e) {
            return back()->with('error', 'حدث خطأ أثناء حذف شريحة الضريبة: ' . $e->getMessage());
        }
    }

    /**
     * Calculate tax for a given amount.
     *
     * @param  float  $amount
     * @return float
     */
    public function calculateTax($amount)
    {
        if ($amount <= $this->min_amount) {
            return 0;
        }

        if ($amount > $this->max_amount && $this->max_amount > 0) {
            $taxableAmount = $this->max_amount - $this->min_amount;
        } else {
            $taxableAmount = $amount - $this->min_amount;
        }

        return $taxableAmount * ($this->tax_rate / 100);
    }
}
