<?php

namespace App\Http\Controllers;

use App\Models\Supplier;
use App\Models\Company;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;

class SupplierController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        $user = Auth::user();
        $query = Supplier::query();

        if (!$user->isSystemAdmin()) {
            $company = $user->company;
            if (!$company) {
                return redirect()->route('dashboard')->with('error', 'لا يوجد شركة مرتبطة بالمستخدم الحالي');
            }
            $query->where('company_id', $company->id);
        } elseif ($request->has('company_id')) {
            $query->where('company_id', $request->company_id);
        }

        $suppliers = $query->orderBy('name')->get();

        return view('inventory.suppliers.index', compact('suppliers'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create(Request $request)
    {
        $user = Auth::user();
        $companyId = $user->isSystemAdmin() ? ($request->company_id ?? $user->company_id) : $user->company_id;

        if ($user->isSystemAdmin() && !$companyId) {
            $companies = Company::all();
            return view('inventory.suppliers.create', compact('companies'));
        }

        return view('inventory.suppliers.create', compact('companyId'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $user = Auth::user();
        $companyId = $user->isSystemAdmin() ? ($request->company_id ?? $user->company_id) : $user->company_id;

        if (!$companyId) {
            return redirect()->back()->with('error', 'يجب تحديد شركة أولاً.')->withInput();
        }

        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'code' => [
                'required',
                'string',
                'max:50',
                // Validar que el código sea único dentro de la misma compañía
                function ($attribute, $value, $fail) use ($companyId) {
                    $exists = Supplier::where('code', $value)
                        ->where('company_id', $companyId)
                        ->exists();

                    if ($exists) {
                        $fail('كود المورد مستخدم بالفعل في هذه الشركة.');
                    }
                },
            ],
            'contact_person' => 'nullable|string|max:255',
            'phone' => 'nullable|string|max:20',
            'email' => 'nullable|email|max:255',
            'website' => 'nullable|url|max:255',
            'address' => 'nullable|string',
            'tax_number' => 'nullable|string|max:50',
            'commercial_register' => 'nullable|string|max:50',
            'is_active' => 'boolean',
            'notes' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $supplier = new Supplier();
        $supplier->company_id = $companyId;
        $supplier->name = $request->name;
        $supplier->code = $request->code;
        $supplier->contact_person = $request->contact_person;
        $supplier->phone = $request->phone;
        $supplier->email = $request->email;
        $supplier->website = $request->website;
        $supplier->address = $request->address;
        $supplier->tax_number = $request->tax_number;
        $supplier->commercial_register = $request->commercial_register;
        $supplier->is_active = $request->has('is_active') ? true : false;
        $supplier->notes = $request->notes;
        $supplier->save();

        return redirect()->route('inventory.suppliers.index', ['company_id' => $companyId])
            ->with('success', 'تم إنشاء المورد بنجاح');
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $user = Auth::user();
        $query = Supplier::with(['evaluations' => function ($query) {
            $query->orderBy('evaluation_date', 'desc');
        }]);

        if (!$user->isSystemAdmin()) {
            $company = $user->company;
            if (!$company) {
                return redirect()->route('dashboard')->with('error', 'لا يوجد شركة مرتبطة بالمستخدم الحالي');
            }
            $query->where('company_id', $company->id);
        }

        $supplier = $query->where('id', $id)->firstOrFail();

        // Get purchase orders for this supplier
        $purchaseOrders = $supplier->purchaseOrders()
            ->orderBy('order_date', 'desc')
            ->limit(10)
            ->get();

        return view('inventory.suppliers.show', compact('supplier', 'purchaseOrders'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $user = Auth::user();
        $query = Supplier::query();

        if (!$user->isSystemAdmin()) {
            $company = $user->company;
            if (!$company) {
                return redirect()->route('dashboard')->with('error', 'لا يوجد شركة مرتبطة بالمستخدم الحالي');
            }
            $query->where('company_id', $company->id);
        }

        $supplier = $query->where('id', $id)->firstOrFail();

        return view('inventory.suppliers.edit', compact('supplier'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $user = Auth::user();
        $query = Supplier::query();

        if (!$user->isSystemAdmin()) {
            $company = $user->company;
            if (!$company) {
                return redirect()->route('dashboard')->with('error', 'لا يوجد شركة مرتبطة بالمستخدم الحالي');
            }
            $query->where('company_id', $company->id);
        }

        $supplier = $query->where('id', $id)->firstOrFail();
        $companyId = $supplier->company_id;

        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'code' => [
                'required',
                'string',
                'max:50',
                // Validar que el código sea único dentro de la misma compañía, excluyendo el proveedor actual
                function ($attribute, $value, $fail) use ($companyId, $supplier) {
                    $exists = Supplier::where('code', $value)
                        ->where('company_id', $companyId)
                        ->where('id', '!=', $supplier->id)
                        ->exists();

                    if ($exists) {
                        $fail('كود المورد مستخدم بالفعل في هذه الشركة.');
                    }
                },
            ],
            'contact_person' => 'nullable|string|max:255',
            'phone' => 'nullable|string|max:20',
            'email' => 'nullable|email|max:255',
            'website' => 'nullable|url|max:255',
            'address' => 'nullable|string',
            'tax_number' => 'nullable|string|max:50',
            'commercial_register' => 'nullable|string|max:50',
            'is_active' => 'boolean',
            'notes' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        $supplier->name = $request->name;
        $supplier->code = $request->code;
        $supplier->contact_person = $request->contact_person;
        $supplier->phone = $request->phone;
        $supplier->email = $request->email;
        $supplier->website = $request->website;
        $supplier->address = $request->address;
        $supplier->tax_number = $request->tax_number;
        $supplier->commercial_register = $request->commercial_register;
        $supplier->is_active = $request->has('is_active') ? true : false;
        $supplier->notes = $request->notes;
        $supplier->save();

        return redirect()->route('inventory.suppliers.index', ['company_id' => $companyId])
            ->with('success', 'تم تحديث المورد بنجاح');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        $user = Auth::user();
        $query = Supplier::query();

        if (!$user->isSystemAdmin()) {
            $company = $user->company;
            if (!$company) {
                return redirect()->route('dashboard')->with('error', 'لا يوجد شركة مرتبطة بالمستخدم الحالي');
            }
            $query->where('company_id', $company->id);
        }

        $supplier = $query->where('id', $id)->firstOrFail();
        $companyId = $supplier->company_id;

        // Check if supplier has any purchase orders
        $hasPurchaseOrders = $supplier->purchaseOrders()->count() > 0;

        if ($hasPurchaseOrders) {
            return redirect()->back()
                ->with('error', 'لا يمكن حذف المورد لأنه مرتبط بأوامر شراء');
        }

        $supplier->delete();

        return redirect()->route('inventory.suppliers.index', ['company_id' => $companyId])
            ->with('success', 'تم حذف المورد بنجاح');
    }
}
