<?php

namespace App\Http\Controllers;

use App\Models\SalesReturn;
use App\Models\SalesReturnItem;
use App\Models\Customer;
use App\Models\Invoice;
use App\Models\Warehouse;
use App\Models\InventoryItem;
use App\Services\AccountingService;
use App\Services\InventoryService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;
use Illuminate\Validation\Rule;

class SalesReturnController extends Controller
{
    protected $accountingService;
    protected $inventoryService;

    public function __construct(AccountingService $accountingService, InventoryService $inventoryService)
    {
        $this->accountingService = $accountingService;
        $this->inventoryService = $inventoryService;
        $this->middleware('auth');
    }

    /**
     * عرض قائمة مرتجعات المبيعات
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        $user = auth()->user();
        $query = SalesReturn::with(['customer', 'warehouse', 'creator']);

        if ($user->isSystemAdmin()) {
            if ($request->has('company_id')) {
                $query->where('company_id', $request->company_id);
            }
        } else {
            $query->where('company_id', $user->company_id);
        }

        $salesReturns = $query->orderBy('created_at', 'desc')
            ->paginate(10);

        return view('sales_returns.index', compact('salesReturns'));
    }

    /**
     * عرض نموذج إنشاء مرتجع مبيعات جديد
     *
     * @return \Illuminate\Http\Response
     */
    public function create(Request $request)
    {
        $user = auth()->user();
        $companyId = $user->isSystemAdmin() ? ($request->company_id ?? $user->company_id) : $user->company_id;

        if ($user->isSystemAdmin() && !$companyId) {
            $companies = \App\Models\Company::all();
            return view('sales_returns.create', compact('companies'));
        }

        $customers = Customer::where('company_id', $companyId)->get();
        $warehouses = Warehouse::where('company_id', $companyId)->where('is_active', true)->get();
        $invoices = Invoice::where('company_id', $companyId)
            ->where('status', 'paid')
            ->orderBy('created_at', 'desc')
            ->get();
        $items = InventoryItem::where('company_id', $companyId)->where('is_active', true)->get();

        // توليد رقم مرتجع جديد
        $lastReturn = SalesReturn::where('company_id', $companyId)
            ->orderBy('id', 'desc')
            ->first();
        $returnNumber = 'SR-' . date('Ymd') . '-' . sprintf('%04d', ($lastReturn ? ($lastReturn->id + 1) : 1));

        return view('sales_returns.create', compact('customers', 'warehouses', 'invoices', 'items', 'returnNumber', 'companyId'));
    }

    /**
     * تخزين مرتجع مبيعات جديد
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $user = auth()->user();
        $validator = Validator::make($request->all(), [
            'customer_id' => 'required|exists:customers,id',
            'invoice_id' => 'nullable|exists:invoices,id',
            'warehouse_id' => 'required|exists:warehouses,id',
            'return_number' => 'required|string',
            'return_date' => 'required|date',
            'notes' => 'nullable|string',
            'internal_notes' => 'nullable|string',
            'items' => 'required|array|min:1',
            'items.*.item_id' => 'required|exists:inventory_items,id',
            'items.*.invoice_item_id' => 'nullable|exists:invoice_items,id',
            'items.*.quantity' => 'required|numeric|min:0.01',
            'items.*.unit_price' => 'required|numeric|min:0',
            'items.*.tax_rate' => 'nullable|numeric|min:0',
            'items.*.discount_amount' => 'nullable|numeric|min:0',
        ]);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        // Get customer to derive company_id
        $customer = Customer::findOrFail($request->customer_id);
        $companyId = $customer->company_id;

        // Check if return number is unique for this company
        $existing = SalesReturn::where('company_id', $companyId)
            ->where('return_number', $request->return_number)
            ->first();
        if ($existing) {
            return redirect()->back()
                ->with('error', 'رقم المرتجع مستخدم بالفعل لهذه الشركة.')
                ->withInput();
        }

        DB::beginTransaction();

        try {
            // حساب المجاميع
            $subtotal = 0;
            $taxAmount = 0;
            $discountAmount = 0;
            $totalAmount = 0;

            foreach ($request->items as $item) {
                $itemSubtotal = $item['quantity'] * $item['unit_price'];
                $itemTaxAmount = $itemSubtotal * ($item['tax_rate'] ?? 0) / 100;
                $itemDiscountAmount = $item['discount_amount'] ?? 0;
                $itemTotal = $itemSubtotal + $itemTaxAmount - $itemDiscountAmount;

                $subtotal += $itemSubtotal;
                $taxAmount += $itemTaxAmount;
                $discountAmount += $itemDiscountAmount;
                $totalAmount += $itemTotal;
            }

            // إنشاء مرتجع المبيعات
            $salesReturn = SalesReturn::create([
                'company_id' => $companyId,
                'customer_id' => $request->customer_id,
                'invoice_id' => $request->invoice_id,
                'warehouse_id' => $request->warehouse_id,
                'created_by' => auth()->id(),
                'return_number' => $request->return_number,
                'return_date' => $request->return_date,
                'subtotal' => $subtotal,
                'tax_amount' => $taxAmount,
                'discount_amount' => $discountAmount,
                'total_amount' => $totalAmount,
                'status' => 'draft',
                'notes' => $request->notes,
                'internal_notes' => $request->internal_notes,
            ]);

            // إنشاء بنود مرتجع المبيعات
            foreach ($request->items as $item) {
                $itemSubtotal = $item['quantity'] * $item['unit_price'];
                $itemTaxAmount = $itemSubtotal * ($item['tax_rate'] ?? 0) / 100;
                $itemDiscountAmount = $item['discount_amount'] ?? 0;
                $itemTotal = $itemSubtotal + $itemTaxAmount - $itemDiscountAmount;

                SalesReturnItem::create([
                    'sales_return_id' => $salesReturn->id,
                    'item_id' => $item['item_id'],
                    'invoice_item_id' => $item['invoice_item_id'] ?? null,
                    'quantity' => $item['quantity'],
                    'unit_price' => $item['unit_price'],
                    'tax_rate' => $item['tax_rate'] ?? 0,
                    'tax_amount' => $itemTaxAmount,
                    'discount_amount' => $itemDiscountAmount,
                    'total_amount' => $itemTotal,
                    'notes' => $item['notes'] ?? null,
                ]);
            }

            DB::commit();

            return redirect()->route('sales-returns.show', $salesReturn->id)
                ->with('success', 'تم إنشاء مرتجع المبيعات بنجاح');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()
                ->withInput()
                ->with('error', 'حدث خطأ أثناء إنشاء مرتجع المبيعات: ' . $e->getMessage());
        }
    }

    /**
     * عرض مرتجع مبيعات محدد
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $user = auth()->user();
        $query = SalesReturn::with(['customer', 'warehouse', 'invoice', 'creator', 'approver', 'items.item']);

        if (!$user->isSystemAdmin()) {
            $query->where('company_id', $user->company_id);
        }

        $salesReturn = $query->findOrFail($id);
        return view('sales_returns.show', compact('salesReturn'));
    }

    /**
     * عرض نموذج تعديل مرتجع مبيعات
     *
     * @param  \App\Models\SalesReturn  $salesReturn
     * @return \Illuminate\Http\Response
     */
    public function edit(SalesReturn $salesReturn)
    {
        $user = auth()->user();
        if (!$user->isSystemAdmin() && $salesReturn->company_id !== $user->company_id) {
            abort(403);
        }

        if ($salesReturn->status !== 'draft') {
            return redirect()->route('sales-returns.show', $salesReturn)
                ->with('error', 'لا يمكن تعديل مرتجع المبيعات بعد اعتماده');
        }

        $companyId = $salesReturn->company_id;
        $customers = Customer::where('company_id', $companyId)->get();
        $warehouses = Warehouse::where('company_id', $companyId)->where('is_active', true)->get();
        $invoices = Invoice::where('company_id', $companyId)
            ->where('status', 'paid')
            ->orderBy('created_at', 'desc')
            ->get();
        $items = InventoryItem::where('company_id', $companyId)->where('is_active', true)->get();

        $salesReturn->load(['items.item', 'items.invoiceItem']);

        return view('sales_returns.edit', compact('salesReturn', 'customers', 'warehouses', 'invoices', 'items'));
    }

    /**
     * تحديث مرتجع مبيعات محدد
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\SalesReturn  $salesReturn
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, SalesReturn $salesReturn)
    {
        $user = auth()->user();
        if (!$user->isSystemAdmin() && $salesReturn->company_id !== $user->company_id) {
            abort(403);
        }

        if ($salesReturn->status !== 'draft') {
            return redirect()->route('sales-returns.show', $salesReturn)
                ->with('error', 'لا يمكن تعديل مرتجع المبيعات بعد اعتماده');
        }

        // Get customer to derive company_id
        $customer = Customer::findOrFail($request->customer_id);
        $companyId = $customer->company_id;

        $validated = $request->validate([
            'customer_id' => 'required|exists:customers,id',
            'invoice_id' => 'nullable|exists:invoices,id',
            'warehouse_id' => 'required|exists:warehouses,id',
            'return_number' => [
                'required',
                Rule::unique('sales_returns')->where(function ($query) use ($companyId) {
                    return $query->where('company_id', $companyId);
                })->ignore($salesReturn->id),
            ],
            'return_date' => 'required|date',
            'notes' => 'nullable|string',
            'internal_notes' => 'nullable|string',
            'items' => 'required|array|min:1',
            'items.*.id' => 'nullable|exists:sales_return_items,id',
            'items.*.item_id' => 'required|exists:inventory_items,id',
            'items.*.invoice_item_id' => 'nullable|exists:invoice_items,id',
            'items.*.quantity' => 'required|numeric|min:0.01',
            'items.*.unit_price' => 'required|numeric|min:0',
            'items.*.tax_rate' => 'nullable|numeric|min:0',
            'items.*.discount_amount' => 'nullable|numeric|min:0',
        ]);

        DB::beginTransaction();

        try {
            // حساب المجاميع
            $subtotal = 0;
            $taxAmount = 0;
            $discountAmount = 0;
            $totalAmount = 0;

            foreach ($validated['items'] as $item) {
                $itemSubtotal = $item['quantity'] * $item['unit_price'];
                $itemTaxAmount = $itemSubtotal * ($item['tax_rate'] ?? 0) / 100;
                $itemDiscountAmount = $item['discount_amount'] ?? 0;
                $itemTotal = $itemSubtotal + $itemTaxAmount - $itemDiscountAmount;

                $subtotal += $itemSubtotal;
                $taxAmount += $itemTaxAmount;
                $discountAmount += $itemDiscountAmount;
                $totalAmount += $itemTotal;
            }

            // تحديث مرتجع المبيعات
            $salesReturn->update([
                'company_id' => $companyId,
                'customer_id' => $validated['customer_id'],
                'invoice_id' => $validated['invoice_id'],
                'warehouse_id' => $validated['warehouse_id'],
                'return_number' => $validated['return_number'],
                'return_date' => $validated['return_date'],
                'subtotal' => $subtotal,
                'tax_amount' => $taxAmount,
                'discount_amount' => $discountAmount,
                'total_amount' => $totalAmount,
                'notes' => $validated['notes'],
                'internal_notes' => $validated['internal_notes'],
            ]);

            // حذف البنود القديمة غير الموجودة في الطلب الجديد
            $existingItemIds = collect($validated['items'])->pluck('id')->filter()->toArray();
            $salesReturn->items()->whereNotIn('id', $existingItemIds)->delete();

            // إنشاء أو تحديث البنود
            foreach ($validated['items'] as $item) {
                $itemSubtotal = $item['quantity'] * $item['unit_price'];
                $itemTaxAmount = $itemSubtotal * ($item['tax_rate'] ?? 0) / 100;
                $itemDiscountAmount = $item['discount_amount'] ?? 0;
                $itemTotal = $itemSubtotal + $itemTaxAmount - $itemDiscountAmount;

                $itemData = [
                    'sales_return_id' => $salesReturn->id,
                    'item_id' => $item['item_id'],
                    'invoice_item_id' => $item['invoice_item_id'] ?? null,
                    'quantity' => $item['quantity'],
                    'unit_price' => $item['unit_price'],
                    'tax_rate' => $item['tax_rate'] ?? 0,
                    'tax_amount' => $itemTaxAmount,
                    'discount_amount' => $itemDiscountAmount,
                    'total_amount' => $itemTotal,
                    'notes' => $item['notes'] ?? null,
                ];

                if (isset($item['id']) && $item['id']) {
                    SalesReturnItem::where('id', $item['id'])->update($itemData);
                } else {
                    SalesReturnItem::create($itemData);
                }
            }

            DB::commit();

            return redirect()->route('sales-returns.show', $salesReturn)
                ->with('success', 'تم تحديث مرتجع المبيعات بنجاح');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()
                ->withInput()
                ->with('error', 'حدث خطأ أثناء تحديث مرتجع المبيعات: ' . $e->getMessage());
        }
    }

    /**
     * اعتماد مرتجع مبيعات
     *
     * @param  \App\Models\SalesReturn  $salesReturn
     * @return \Illuminate\Http\Response
     */
    public function approve(SalesReturn $salesReturn)
    {
        $user = auth()->user();
        if (!$user->isSystemAdmin() && $salesReturn->company_id !== $user->company_id) {
            abort(403);
        }

        if ($salesReturn->status !== 'draft') {
            return redirect()->route('sales-returns.show', $salesReturn)
                ->with('error', 'تم اعتماد مرتجع المبيعات مسبقاً');
        }

        DB::beginTransaction();

        try {
            // تحديث حالة مرتجع المبيعات
            $salesReturn->update([
                'status' => 'approved',
                'approved_by' => Auth::id(),
                'approved_at' => now(),
            ]);

            // تحديث المخزون
            $this->inventoryService->updateInventoryForSalesReturn($salesReturn);

            // إنشاء القيود المحاسبية
            $this->createAccountingEntries($salesReturn);

            DB::commit();

            return redirect()->route('sales-returns.show', $salesReturn)
                ->with('success', 'تم اعتماد مرتجع المبيعات بنجاح');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()
                ->with('error', 'حدث خطأ أثناء اعتماد مرتجع المبيعات: ' . $e->getMessage());
        }
    }

    /**
     * إلغاء مرتجع مبيعات
     *
     * @param  \App\Models\SalesReturn  $salesReturn
     * @return \Illuminate\Http\Response
     */
    public function cancel(SalesReturn $salesReturn)
    {
        $user = auth()->user();
        if (!$user->isSystemAdmin() && $salesReturn->company_id !== $user->company_id) {
            abort(403);
        }

        if ($salesReturn->status === 'cancelled') {
            return redirect()->route('sales-returns.show', $salesReturn)
                ->with('error', 'تم إلغاء مرتجع المبيعات مسبقاً');
        }

        DB::beginTransaction();

        try {
            // تحديث حالة مرتجع المبيعات
            $salesReturn->update([
                'status' => 'cancelled',
            ]);

            // إلغاء تحديث المخزون إذا كان المرتجع معتمداً
            if ($salesReturn->approved_at) {
                $this->inventoryService->reverseInventoryForSalesReturn($salesReturn);
            }

            // إلغاء القيود المحاسبية
            $this->cancelAccountingEntries($salesReturn);

            DB::commit();

            return redirect()->route('sales-returns.show', $salesReturn)
                ->with('success', 'تم إلغاء مرتجع المبيعات بنجاح');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()
                ->with('error', 'حدث خطأ أثناء إلغاء مرتجع المبيعات: ' . $e->getMessage());
        }
    }

    /**
     * حذف مرتجع مبيعات
     *
     * @param  \App\Models\SalesReturn  $salesReturn
     * @return \Illuminate\Http\Response
     */
    public function destroy(SalesReturn $salesReturn)
    {
        $user = auth()->user();
        if (!$user->isSystemAdmin() && $salesReturn->company_id !== $user->company_id) {
            abort(403);
        }

        if ($salesReturn->status !== 'draft') {
            return redirect()->route('sales-returns.show', $salesReturn)
                ->with('error', 'لا يمكن حذف مرتجع المبيعات بعد اعتماده');
        }

        DB::beginTransaction();

        try {
            // حذف بنود مرتجع المبيعات
            $salesReturn->items()->delete();

            // حذف مرتجع المبيعات
            $salesReturn->delete();

            DB::commit();

            return redirect()->route('sales-returns.index')
                ->with('success', 'تم حذف مرتجع المبيعات بنجاح');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()
                ->with('error', 'حدث خطأ أثناء حذف مرتجع المبيعات: ' . $e->getMessage());
        }
    }

    /**
     * إنشاء القيود المحاسبية لمرتجع المبيعات
     *
     * @param  \App\Models\SalesReturn  $salesReturn
     * @return void
     */
    private function createAccountingEntries(SalesReturn $salesReturn)
    {
        // التأكد من وجود الحسابات الرئيسية
        $accounts = $this->accountingService->ensureParentAccounts($salesReturn->company_id);

        // الحصول على حساب العميل
        $customer = Customer::find($salesReturn->customer_id);
        $customerAccount = $this->accountingService->ensureCustomerAccount(
            $salesReturn->company_id,
            $salesReturn->customer_id,
            $customer->name
        );

        // إنشاء قيد محاسبي لمرتجع المبيعات
        $journalEntryData = [
            'company_id' => $salesReturn->company_id,
            'entry_date' => $salesReturn->return_date,
            'reference_number' => $salesReturn->return_number,
            'reference_type' => 'sales_return',
            'description' => 'قيد محاسبي لمرتجع المبيعات رقم ' . $salesReturn->return_number,
            'created_by' => Auth::id(),
            'status' => 'approved',
            'journalable_type' => SalesReturn::class,
            'journalable_id' => $salesReturn->id,
            'items' => []
        ];

        // إضافة بنود القيد المحاسبي

        // 1. مدين: حساب المخزون (بقيمة تكلفة البضاعة)
        $journalEntryData['items'][] = [
            'account_id' => $accounts['inventory']->id,
            'description' => 'إضافة للمخزون من مرتجع المبيعات رقم ' . $salesReturn->return_number,
            'debit' => $salesReturn->subtotal,
            'credit' => 0
        ];

        // 2. مدين: حساب مرتجع المبيعات (بقيمة المبيعات)
        $journalEntryData['items'][] = [
            'account_id' => $accounts['sales_returns']->id,
            'description' => 'مرتجع المبيعات رقم ' . $salesReturn->return_number,
            'debit' => $salesReturn->subtotal,
            'credit' => 0
        ];

        // 3. دائن: حساب العميل (بالمبلغ الإجمالي)
        $journalEntryData['items'][] = [
            'account_id' => $customerAccount->id,
            'description' => 'مرتجع المبيعات للعميل ' . $customer->name . ' رقم ' . $salesReturn->return_number,
            'debit' => 0,
            'credit' => $salesReturn->total_amount
        ];

        // 4. مدين: حساب ضريبة القيمة المضافة (إذا كان هناك ضريبة)
        if ($salesReturn->tax_amount > 0) {
            $journalEntryData['items'][] = [
                'account_id' => $accounts['vat']->id,
                'description' => 'ضريبة القيمة المضافة لمرتجع المبيعات رقم ' . $salesReturn->return_number,
                'debit' => $salesReturn->tax_amount,
                'credit' => 0
            ];
        }

        // إنشاء القيد المحاسبي
        $this->accountingService->createJournalEntry($journalEntryData);
    }

    /**
     * إلغاء القيود المحاسبية لمرتجع المبيعات
     *
     * @param  \App\Models\SalesReturn  $salesReturn
     * @return void
     */
    private function cancelAccountingEntries(SalesReturn $salesReturn)
    {
        // حذف القيود المحاسبية السابقة
        $salesReturn->journalEntries()->delete();

        // إذا كان المرتجع معتمداً، إنشاء قيد عكسي
        if ($salesReturn->approved_at) {
            // التأكد من وجود الحسابات الرئيسية
            $accounts = $this->accountingService->ensureParentAccounts($salesReturn->company_id);

            // الحصول على حساب العميل
            $customer = Customer::find($salesReturn->customer_id);
            $customerAccount = $this->accountingService->ensureCustomerAccount(
                $salesReturn->company_id,
                $salesReturn->customer_id,
                $customer->name
            );

            // إنشاء قيد محاسبي عكسي لمرتجع المبيعات
            $journalEntryData = [
                'company_id' => $salesReturn->company_id,
                'entry_date' => now(),
                'reference_number' => $salesReturn->return_number . '-CANCEL',
                'reference_type' => 'sales_return_cancel',
                'description' => 'قيد محاسبي لإلغاء مرتجع المبيعات رقم ' . $salesReturn->return_number,
                'created_by' => Auth::id(),
                'status' => 'approved',
                'journalable_type' => SalesReturn::class,
                'journalable_id' => $salesReturn->id,
                'items' => []
            ];

            // إضافة بنود القيد المحاسبي (عكس القيد الأصلي)

            // 1. دائن: حساب المخزون (بقيمة تكلفة البضاعة)
            $journalEntryData['items'][] = [
                'account_id' => $accounts['inventory']->id,
                'description' => 'إلغاء إضافة للمخزون من مرتجع المبيعات رقم ' . $salesReturn->return_number,
                'debit' => 0,
                'credit' => $salesReturn->subtotal
            ];

            // 2. دائن: حساب مرتجع المبيعات (بقيمة المبيعات)
            $journalEntryData['items'][] = [
                'account_id' => $accounts['sales_returns']->id,
                'description' => 'إلغاء مرتجع المبيعات رقم ' . $salesReturn->return_number,
                'debit' => 0,
                'credit' => $salesReturn->subtotal
            ];

            // 3. مدين: حساب العميل (بالمبلغ الإجمالي)
            $journalEntryData['items'][] = [
                'account_id' => $customerAccount->id,
                'description' => 'إلغاء مرتجع المبيعات للعميل ' . $customer->name . ' رقم ' . $salesReturn->return_number,
                'debit' => $salesReturn->total_amount,
                'credit' => 0
            ];

            // 4. دائن: حساب ضريبة القيمة المضافة (إذا كان هناك ضريبة)
            if ($salesReturn->tax_amount > 0) {
                $journalEntryData['items'][] = [
                    'account_id' => $accounts['vat']->id,
                    'description' => 'إلغاء ضريبة القيمة المضافة لمرتجع المبيعات رقم ' . $salesReturn->return_number,
                    'debit' => 0,
                    'credit' => $salesReturn->tax_amount
                ];
            }

            // إنشاء القيد المحاسبي
            $this->accountingService->createJournalEntry($journalEntryData);
        }
    }

    /**
     * الحصول على بنود الفاتورة
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function getInvoiceItems(Request $request)
    {
        $user = auth()->user();
        $invoiceId = $request->input('invoice_id');
        $query = Invoice::with(['items.item']);

        if (!$user->isSystemAdmin()) {
            $query->where('company_id', $user->company_id);
        }

        $invoice = $query->findOrFail($invoiceId);

        return response()->json([
            'items' => $invoice->items,
            'customer_id' => $invoice->customer_id
        ]);
    }
}
