<?php

namespace App\Http\Controllers;

use App\Models\Company;
use App\Models\SalaryPolicy;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class SalaryPolicyController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        $companyId = $request->input('company_id', Auth::user()->company_id);

        $salaryPolicies = SalaryPolicy::where('company_id', $companyId)
            ->orderBy('created_at', 'desc')
            ->paginate(10);

        return view('hrm.salary_policies.index', compact('salaryPolicies'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $companies = Company::all();

        return view('hrm.salary_policies.create', compact('companies'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $request->validate([
            'company_id' => 'required|exists:companies,id',
            'name' => 'required|string|max:255',
            'tax_calculation_method' => 'required|in:progressive,flat',
            'social_insurance_percentage' => 'required|numeric|min:0|max:100',
            'overtime_rate' => 'required|numeric|min:0',
            'late_deduction_rate' => 'required|numeric|min:0',
            'absence_deduction_method' => 'required|in:daily,hourly',
            'calculate_overtime' => 'boolean',
        ]);

        try {
            $salaryPolicy = new SalaryPolicy();
            $salaryPolicy->company_id = $request->company_id;
            $salaryPolicy->name = $request->name;
            $salaryPolicy->tax_calculation_method = $request->tax_calculation_method;
            $salaryPolicy->social_insurance_percentage = $request->social_insurance_percentage;
            $salaryPolicy->overtime_rate = $request->overtime_rate;
            $salaryPolicy->late_deduction_rate = $request->late_deduction_rate;
            $salaryPolicy->absence_deduction_method = $request->absence_deduction_method;
            $salaryPolicy->calculate_overtime = $request->has('calculate_overtime');
            $salaryPolicy->created_by = Auth::id();
            $salaryPolicy->save();

            return redirect()->route('hrm.salary-policies.index')
                ->with('success', 'تم إنشاء سياسة الرواتب بنجاح');
        } catch (\Exception $e) {
            return back()->with('error', 'حدث خطأ أثناء إنشاء سياسة الرواتب: ' . $e->getMessage());
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $salaryPolicy = SalaryPolicy::findOrFail($id);

        // Check if user has access to this salary policy
        if (Auth::user()->company_id != $salaryPolicy->company_id && !Auth::user()->hasRole('super-admin')) {
            return redirect()->route('hrm.salary-policies.index')
                ->with('error', 'ليس لديك صلاحية للوصول إلى سياسة الرواتب هذه');
        }

        return view('hrm.salary_policies.show', compact('salaryPolicy'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $salaryPolicy = SalaryPolicy::findOrFail($id);

        // Check if user has access to this salary policy
        if (Auth::user()->company_id != $salaryPolicy->company_id && !Auth::user()->hasRole('super-admin')) {
            return redirect()->route('hrm.salary-policies.index')
                ->with('error', 'ليس لديك صلاحية لتعديل سياسة الرواتب هذه');
        }

        $companies = Company::all();

        return view('hrm.salary_policies.edit', compact('salaryPolicy', 'companies'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'tax_calculation_method' => 'required|in:progressive,flat',
            'social_insurance_percentage' => 'required|numeric|min:0|max:100',
            'overtime_rate' => 'required|numeric|min:0',
            'late_deduction_rate' => 'required|numeric|min:0',
            'absence_deduction_method' => 'required|in:daily,hourly',
            'calculate_overtime' => 'boolean',
        ]);

        $salaryPolicy = SalaryPolicy::findOrFail($id);

        // Check if user has access to this salary policy
        if (Auth::user()->company_id != $salaryPolicy->company_id && !Auth::user()->hasRole('super-admin')) {
            return redirect()->route('hrm.salary-policies.index')
                ->with('error', 'ليس لديك صلاحية لتعديل سياسة الرواتب هذه');
        }

        try {
            $salaryPolicy->name = $request->name;
            $salaryPolicy->tax_calculation_method = $request->tax_calculation_method;
            $salaryPolicy->social_insurance_percentage = $request->social_insurance_percentage;
            $salaryPolicy->overtime_rate = $request->overtime_rate;
            $salaryPolicy->late_deduction_rate = $request->late_deduction_rate;
            $salaryPolicy->absence_deduction_method = $request->absence_deduction_method;
            $salaryPolicy->calculate_overtime = $request->has('calculate_overtime');
            $salaryPolicy->updated_by = Auth::id();
            $salaryPolicy->save();

            return redirect()->route('hrm.salary-policies.show', $salaryPolicy->id)
                ->with('success', 'تم تحديث سياسة الرواتب بنجاح');
        } catch (\Exception $e) {
            return back()->with('error', 'حدث خطأ أثناء تحديث سياسة الرواتب: ' . $e->getMessage());
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        $salaryPolicy = SalaryPolicy::findOrFail($id);

        // Check if user has access to this salary policy
        if (Auth::user()->company_id != $salaryPolicy->company_id && !Auth::user()->hasRole('super-admin')) {
            return redirect()->route('hrm.salary-policies.index')
                ->with('error', 'ليس لديك صلاحية لحذف سياسة الرواتب هذه');
        }

        try {
            $salaryPolicy->delete();

            return redirect()->route('hrm.salary-policies.index')
                ->with('success', 'تم حذف سياسة الرواتب بنجاح');
        } catch (\Exception $e) {
            return back()->with('error', 'حدث خطأ أثناء حذف سياسة الرواتب: ' . $e->getMessage());
        }
    }
}
