<?php

namespace App\Http\Controllers;

use App\Models\Permission;
use App\Models\Role;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Illuminate\Validation\Rule;

class RoleController extends Controller
{
    /**
     * إنشاء مثيل جديد من وحدة التحكم.
     *
     * @return void
     */
    public function __construct()
    {
        $this->middleware('auth');
        // تعديل الصلاحية للسماح بالوصول للمدير العام والمدير
        $this->middleware(function ($request, $next) {
            // التحقق من الدور باستخدام الاسم أو الـ slug
            if (auth()->user()->hasAnyRole(['super_admin', 'admin', 'مدير النظام'])) {
                return $next($request);
            }

            if (auth()->user()->hasPermissionTo('manage_roles')) {
                return $next($request);
            }

            abort(403, 'ليس لديك صلاحية للوصول إلى هذه الصفحة');
        });
    }

    /**
     * عرض قائمة الأدوار.
     *
     * @return \Illuminate\View\View
     */
    public function index()
    {
        $query = Role::query()->whereNull('parent_id');

        // تطبيق عزل الشركات - عرض أدوار الشركة الحالية فقط إلا إذا كان المستخدم مدير النظام
        if (!auth()->user()->hasRole('super_admin')) {
            $companyId = auth()->user()->company_id;
            $query->where(function ($q) use ($companyId) {
                $q->whereNull('company_id')  // الأدوار العامة
                    ->orWhere('company_id', $companyId);  // أدوار الشركة الحالية
            });
        }

        $roles = $query->with(['children' => function ($query) use (&$companyId) {
            // تطبيق عزل الشركات على الأدوار الفرعية أيضًا
            if (!auth()->user()->hasRole('super_admin') && isset($companyId)) {
                $query->where(function ($q) use ($companyId) {
                    $q->whereNull('company_id')
                        ->orWhere('company_id', $companyId);
                });
            }
        }])
            ->orderBy('order')
            ->orderBy('name')
            ->get();

        return view('admin.roles.index', compact('roles'));
    }

    /**
     * عرض نموذج إنشاء دور جديد.
     *
     * @return \Illuminate\View\View
     */
    public function create()
    {
        // الحصول على الأدوار المناسبة للشركة
        $rolesQuery = Role::query();

        // إذا لم يكن المستخدم مدير النظام، عرض الأدوار العامة والأدوار الخاصة بالشركة فقط
        if (!auth()->user()->hasRole('super_admin')) {
            $companyId = auth()->user()->company_id;
            $rolesQuery->where(function ($query) use ($companyId) {
                $query->whereNull('company_id')
                    ->orWhere('company_id', $companyId);
            });
        }

        $roles = $rolesQuery->orderBy('name')->get();

        // تحميل جميع الصلاحيات المتاحة
        $permissions = Permission::where('is_active', true)
            ->orderBy('group')
            ->orderBy('name')
            ->get()
            ->groupBy('group');

        return view('admin.roles.create', compact('roles', 'permissions'));
    }

    /**
     * تخزين دور جديد.
     *
     * @param \Illuminate\Http\Request $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'slug' => 'required|string|max:255|unique:roles',
            'description' => 'nullable|string',
            'parent_id' => 'nullable|exists:roles,id',
            'company_id' => 'nullable|exists:companies,id',
            'order' => 'nullable|integer',
            'is_active' => 'boolean',
            'is_default' => 'boolean',
            'permissions' => 'nullable|array',
            'permissions.*' => 'exists:permissions,id',
        ]);

        // التحقق من صلاحية الشركة
        if (
            !auth()->user()->hasRole('super_admin') &&
            $request->company_id !== null &&
            $request->company_id != auth()->user()->company_id
        ) {
            return back()->withInput()->withErrors([
                'company_id' => 'ليس لديك صلاحية لإنشاء دور لهذه الشركة',
            ]);
        }

        // التحقق من صلاحية الدور الأب
        if ($request->parent_id) {
            $parentRole = Role::find($request->parent_id);
            if (
                !auth()->user()->hasRole('super_admin') &&
                $parentRole->company_id !== null &&
                $parentRole->company_id != auth()->user()->company_id
            ) {
                return back()->withInput()->withErrors([
                    'parent_id' => 'ليس لديك صلاحية لاستخدام هذا الدور كأب',
                ]);
            }
        }

        DB::beginTransaction();

        try {
            $role = new Role();
            $role->name = $request->name;
            $role->slug = $request->slug;
            $role->description = $request->description;
            $role->parent_id = $request->parent_id;

            // إذا لم يكن المستخدم مدير النظام، تعيين الشركة الحالية تلقائيًا إذا لم يتم تحديدها
            if (!auth()->user()->hasRole('super_admin') && $request->company_id === null) {
                $role->company_id = auth()->user()->company_id;
            } else {
                $role->company_id = $request->company_id;
            }

            $role->order = $request->order ?? 0;
            $role->is_active = $request->is_active ?? true;
            $role->is_default = $request->is_default ?? false;
            $role->is_system_role = false;
            $role->save();

            // تحديث المسار والمستوى
            $role->updatePath();
            $role->updateLevel();

            // إضافة الصلاحيات
            if ($request->has('permissions')) {
                $role->permissions()->sync($request->permissions);
            }

            DB::commit();

            return redirect()->route('roles.index')
                ->with('success', 'تم إنشاء الدور بنجاح.');
        } catch (\Exception $e) {
            DB::rollBack();

            return back()->withInput()->withErrors([
                'error' => 'حدث خطأ أثناء إنشاء الدور: ' . $e->getMessage(),
            ]);
        }
    }

    /**
     * عرض تفاصيل الدور.
     *
     * @param \App\Models\Role $role
     * @return \Illuminate\View\View
     */
    public function show(Role $role)
    {
        // التحقق من أن الدور ينتمي إلى نفس الشركة أو أنه دور عام
        if (
            !auth()->user()->hasRole('super_admin') &&
            $role->company_id !== null &&
            $role->company_id != auth()->user()->company_id
        ) {
            abort(403, 'ليس لديك صلاحية للوصول إلى هذا الدور');
        }

        $permissions = $role->permissions()->paginate(10);

        // فلترة المستخدمين حسب الشركة الحالية
        $usersQuery = $role->users();
        if (!auth()->user()->hasRole('super_admin')) {
            $companyId = auth()->user()->company_id;
            $usersQuery->where('company_id', $companyId);
        }
        $users = $usersQuery->paginate(10);

        return view('admin.roles.show', compact('role', 'permissions', 'users'));
    }

    /**
     * عرض نموذج تعديل الدور.
     *
     * @param \App\Models\Role $role
     * @return \Illuminate\View\View
     */
    public function edit(Role $role)
    {
        // التحقق من أن الدور ينتمي إلى نفس الشركة أو أنه دور عام
        if (
            !auth()->user()->hasRole('super_admin') &&
            $role->company_id !== null &&
            $role->company_id != auth()->user()->company_id
        ) {
            abort(403, 'ليس لديك صلاحية للوصول إلى هذا الدور');
        }

        // الحصول على الأدوار المناسبة للشركة
        $rolesQuery = Role::where('id', '!=', $role->id);

        // إذا لم يكن المستخدم مدير النظام، عرض الأدوار العامة والأدوار الخاصة بالشركة فقط
        if (!auth()->user()->hasRole('super_admin')) {
            $companyId = auth()->user()->company_id;
            $rolesQuery->where(function ($query) use ($companyId) {
                $query->whereNull('company_id')
                    ->orWhere('company_id', $companyId);
            });
        }

        $roles = $rolesQuery->orderBy('name')->get();

        // تحميل جميع الصلاحيات المتاحة
        $permissions = Permission::where('is_active', true)
            ->orderBy('group')
            ->orderBy('name')
            ->get()
            ->groupBy('group');

        $rolePermissions = $role->permissions->pluck('id')->toArray();

        return view('admin.roles.edit', compact('role', 'roles', 'permissions', 'rolePermissions'));
    }

    /**
     * تحديث الدور.
     *
     * @param \Illuminate\Http\Request $request
     * @param \App\Models\Role $role
     * @return \Illuminate\Http\RedirectResponse
     */
    public function update(Request $request, Role $role)
    {
        // التحقق من أن الدور ينتمي إلى نفس الشركة أو أنه دور عام
        if (
            !auth()->user()->hasRole('super_admin') &&
            $role->company_id !== null &&
            $role->company_id != auth()->user()->company_id
        ) {
            abort(403, 'ليس لديك صلاحية للوصول إلى هذا الدور');
        }

        $request->validate([
            'name' => 'required|string|max:255',
            'slug' => [
                'required',
                'string',
                'max:255',
                Rule::unique('roles')->ignore($role->id),
            ],
            'description' => 'nullable|string',
            'parent_id' => [
                'nullable',
                'exists:roles,id',
                function ($attribute, $value, $fail) use ($role) {
                    if ($value == $role->id) {
                        $fail('لا يمكن تعيين الدور كأب لنفسه.');
                    }

                    // التحقق من عدم تعيين أحد الأبناء كأب
                    $childIds = Role::where('parent_id', $role->id)
                        ->pluck('id')
                        ->toArray();

                    if (in_array($value, $childIds)) {
                        $fail('لا يمكن تعيين أحد الأبناء كأب.');
                    }
                },
            ],
            'company_id' => 'nullable|exists:companies,id',
            'order' => 'nullable|integer',
            'is_active' => 'boolean',
            'is_default' => 'boolean',
            'permissions' => 'nullable|array',
            'permissions.*' => 'exists:permissions,id',
        ]);

        // منع تعديل الأدوار النظامية
        if ($role->is_system_role) {
            return back()->withErrors([
                'error' => 'لا يمكن تعديل الأدوار النظامية.',
            ]);
        }

        // التحقق من صلاحية الشركة
        if (
            !auth()->user()->hasRole('super_admin') &&
            $request->company_id !== null &&
            $request->company_id != auth()->user()->company_id
        ) {
            return back()->withInput()->withErrors([
                'company_id' => 'ليس لديك صلاحية لتعيين الدور لهذه الشركة',
            ]);
        }

        // التحقق من صلاحية الدور الأب
        if ($request->parent_id) {
            $parentRole = Role::find($request->parent_id);
            if (
                !auth()->user()->hasRole('super_admin') &&
                $parentRole->company_id !== null &&
                $parentRole->company_id != auth()->user()->company_id
            ) {
                return back()->withInput()->withErrors([
                    'parent_id' => 'ليس لديك صلاحية لاستخدام هذا الدور كأب',
                ]);
            }
        }

        DB::beginTransaction();

        try {
            $oldParentId = $role->parent_id;

            $role->name = $request->name;
            $role->slug = $request->slug;
            $role->description = $request->description;
            $role->parent_id = $request->parent_id;

            // إذا لم يكن المستخدم مدير النظام، تعيين الشركة الحالية تلقائيًا إذا لم يتم تحديدها
            if (!auth()->user()->hasRole('super_admin') && $request->company_id === null) {
                $role->company_id = auth()->user()->company_id;
            } else {
                $role->company_id = $request->company_id;
            }

            $role->order = $request->order ?? 0;
            $role->is_active = $request->is_active ?? true;
            $role->is_default = $request->is_default ?? false;
            $role->save();

            // تحديث المسار والمستوى إذا تغير الأب
            if ($oldParentId != $request->parent_id) {
                $role->updatePath();
                $role->updateLevel();
            }

            // تحديث الصلاحيات
            try {
                if ($request->has('permissions')) {
                    // تحقق من وجود الصلاحيات قبل المزامنة
                    $validPermissionIds = Permission::whereIn('id', $request->permissions)->pluck('id')->toArray();
                    $role->permissions()->sync($validPermissionIds);
                } else {
                    $role->permissions()->sync([]);
                }
            } catch (\Exception $e) {
                \Log::error('خطأ في مزامنة الصلاحيات: ' . $e->getMessage());
                // استمر في التنفيذ حتى لو فشلت مزامنة الصلاحيات
            }

            DB::commit();

            return redirect()->route('roles.index')
                ->with('success', 'تم تحديث الدور بنجاح.');
        } catch (\Exception $e) {
            DB::rollBack();

            return back()->withInput()->withErrors([
                'error' => 'حدث خطأ أثناء تحديث الدور: ' . $e->getMessage(),
            ]);
        }
    }

    /**
     * حذف الدور.
     *
     * @param \App\Models\Role $role
     * @return \Illuminate\Http\RedirectResponse
     */
    public function destroy(Role $role)
    {
        // التحقق من أن الدور ينتمي إلى نفس الشركة أو أنه دور عام
        if (
            !auth()->user()->hasRole('super_admin') &&
            $role->company_id !== null &&
            $role->company_id != auth()->user()->company_id
        ) {
            abort(403, 'ليس لديك صلاحية للوصول إلى هذا الدور');
        }

        // منع حذف الأدوار النظامية
        if ($role->is_system_role) {
            return back()->withErrors([
                'error' => 'لا يمكن حذف الأدوار النظامية.',
            ]);
        }

        // التحقق من عدم وجود أدوار فرعية
        if ($role->children()->count() > 0) {
            return back()->withErrors([
                'error' => 'لا يمكن حذف الدور لأنه يحتوي على أدوار فرعية.',
            ]);
        }

        DB::beginTransaction();

        try {
            // حذف العلاقات
            $role->permissions()->detach();
            $role->users()->detach();

            // حذف الدور
            $role->delete();

            DB::commit();

            return redirect()->route('roles.index')
                ->with('success', 'تم حذف الدور بنجاح.');
        } catch (\Exception $e) {
            DB::rollBack();

            return back()->withErrors([
                'error' => 'حدث خطأ أثناء حذف الدور: ' . $e->getMessage(),
            ]);
        }
    }

    /**
     * عرض قائمة المستخدمين لإدارة الأدوار.
     *
     * @return \Illuminate\View\View
     */
    public function manageUsers()
    {
        $query = User::query()->orderBy('name');

        // تطبيق عزل الشركات - عرض مستخدمي الشركة الحالية فقط إلا إذا كان المستخدم مدير النظام
        if (!auth()->user()->hasRole('super_admin')) {
            $companyId = auth()->user()->company_id;
            $query->where('company_id', $companyId);
        }

        // إضافة دعم البحث إذا تم تقديم معيار البحث
        if (request()->has('search') && !empty(request('search'))) {
            $searchTerm = request('search');
            $query->where(function ($q) use ($searchTerm) {
                $q->where('name', 'like', "%{$searchTerm}%")
                    ->orWhere('email', 'like', "%{$searchTerm}%");
            });
        }

        $users = $query->paginate(15);

        return view('admin.roles.manage-users', compact('users'));
    }

    /**
     * عرض صفحة إدارة أدوار المستخدم.
     *
     * @param \App\Models\User $user
     * @return \Illuminate\View\View
     */
    public function editUserRoles(User $user)
    {
        // التحقق من أن المستخدم ينتمي إلى نفس الشركة أو أن المستخدم الحالي غير مرتبط بأي شركة
        if (auth()->user()->company_id !== null && $user->company_id != auth()->user()->company_id) {
            abort(403, 'ليس لديك صلاحية للوصول إلى هذا المستخدم');
        }

        // الحصول على الأدوار المناسبة للشركة
        $rolesQuery = Role::query()->orderBy('name');

        // إذا لم يكن المستخدم مدير النظام، عرض الأدوار العامة والأدوار الخاصة بالشركة فقط
        if (!auth()->user()->hasRole('super_admin')) {
            $companyId = auth()->user()->company_id;
            $rolesQuery->where(function ($query) use ($companyId) {
                $query->whereNull('company_id')
                    ->orWhere('company_id', $companyId);
            });
        }

        $roles = $rolesQuery->get();
        $userRoles = $user->roles->pluck('id')->toArray();

        return view('admin.roles.edit-user-roles', compact('user', 'roles', 'userRoles'));
    }

    /**
     * تحديث أدوار المستخدم.
     *
     * @param \Illuminate\Http\Request $request
     * @param \App\Models\User $user
     * @return \Illuminate\Http\RedirectResponse
     */
    public function updateUserRoles(Request $request, User $user)
    {
        // التحقق من أن المستخدم ينتمي إلى نفس الشركة أو أن المستخدم الحالي غير مرتبط بأي شركة
        if (auth()->user()->company_id !== null && $user->company_id != auth()->user()->company_id) {
            abort(403, 'ليس لديك صلاحية للوصول إلى هذا المستخدم');
        }

        $request->validate([
            'roles' => 'nullable|array',
            'roles.*' => 'exists:roles,id',
        ]);

        DB::beginTransaction();

        try {
            // استخدام sync مباشرة بدلاً من syncRoles
            if ($request->has('roles')) {
                $roleIds = $request->roles;

                // التحقق من الأدوار المسموح بها للمستخدم الحالي
                $validRoleIdsQuery = Role::whereIn('id', $roleIds);

                // إذا لم يكن المستخدم مدير النظام، التحقق من أن الأدوار إما عامة أو تنتمي لنفس الشركة
                if (!auth()->user()->hasRole('super_admin')) {
                    $companyId = auth()->user()->company_id;
                    $validRoleIdsQuery->where(function ($query) use ($companyId) {
                        $query->whereNull('company_id')
                            ->orWhere('company_id', $companyId);
                    });
                }

                $validRoleIds = $validRoleIdsQuery->pluck('id')->toArray();
                $user->roles()->sync($validRoleIds);
            } else {
                $user->roles()->sync([]);
            }

            DB::commit();

            return redirect()->route('roles.manage-users')
                ->with('success', 'تم تحديث أدوار المستخدم بنجاح.');
        } catch (\Exception $e) {
            DB::rollBack();

            return back()->withInput()->withErrors([
                'error' => 'حدث خطأ أثناء تحديث أدوار المستخدم: ' . $e->getMessage(),
            ]);
        }
    }
}
