<?php

namespace App\Http\Controllers;

use App\Models\RewardRedemption;
use App\Models\CustomerReward;
use App\Models\Customer;
use App\Models\LoyaltyPoint;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class RewardRedemptionController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        $company_id = $request->input('company_id', Auth::user()->company_id);
        $customer_id = $request->input('customer_id');
        $reward_id = $request->input('reward_id');

        $redemptions = RewardRedemption::with(['customer', 'reward', 'user'])
            ->whereHas('reward', function ($query) use ($company_id) {
                $query->where('company_id', $company_id);
            });

        if ($customer_id) {
            $redemptions->where('customer_id', $customer_id);
        }

        if ($reward_id) {
            $redemptions->where('reward_id', $reward_id);
        }

        $redemptions = $redemptions->orderBy('created_at', 'desc')
            ->paginate(15);

        return view('reward_redemptions.index', compact('redemptions'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create(Request $request)
    {
        $company_id = $request->input('company_id', Auth::user()->company_id);
        $customer_id = $request->input('customer_id');

        $customers = Customer::where('company_id', $company_id)->get();
        $customer = $customer_id ? Customer::findOrFail($customer_id) : null;

        // الحصول على المكافآت النشطة فقط
        $rewards = CustomerReward::where('company_id', $company_id)
            ->where('status', 'active')
            ->where(function ($query) {
                $query->whereNull('end_date')
                    ->orWhere('end_date', '>=', now()->format('Y-m-d'));
            })
            ->where(function ($query) {
                $query->whereNull('usage_limit')
                    ->orWhereRaw('usage_count < usage_limit');
            })
            ->get();

        return view('reward_redemptions.create', compact('customers', 'customer', 'customer_id', 'rewards'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'customer_id' => 'required|exists:customers,id',
            'reward_id' => 'required|exists:customer_rewards,id',
            'notes' => 'nullable|string',
        ]);

        $customer = Customer::findOrFail($validated['customer_id']);
        $reward = CustomerReward::findOrFail($validated['reward_id']);

        // التحقق من أن المستخدم ينتمي إلى نفس الشركة
        if ($reward->company_id != Auth::user()->company_id) {
            abort(403, 'غير مصرح لك باستخدام هذه المكافأة');
        }

        // التحقق من أن المكافأة نشطة
        if ($reward->status != 'active') {
            return redirect()->back()->with('error', 'هذه المكافأة غير نشطة حالياً');
        }

        // التحقق من تاريخ انتهاء المكافأة
        if ($reward->end_date && $reward->end_date < now()->format('Y-m-d')) {
            return redirect()->back()->with('error', 'انتهت صلاحية هذه المكافأة');
        }

        // التحقق من حد الاستخدام
        if ($reward->usage_limit && $reward->usage_count >= $reward->usage_limit) {
            return redirect()->back()->with('error', 'تم الوصول إلى الحد الأقصى لاستخدام هذه المكافأة');
        }

        // التحقق من أن العميل لديه نقاط كافية
        $customerPoints = $customer->getActiveLoyaltyPointsAttribute();
        if ($customerPoints < $reward->points_required) {
            return redirect()->back()->with('error', 'لا يملك العميل نقاط كافية لاستخدام هذه المكافأة');
        }

        DB::beginTransaction();
        try {
            // إنشاء سجل استخدام المكافأة
            $redemption = new RewardRedemption($validated);
            $redemption->user_id = Auth::id();
            $redemption->points_used = $reward->points_required;
            $redemption->save();

            // خصم النقاط من العميل
            $loyaltyPoint = new LoyaltyPoint([
                'company_id' => Auth::user()->company_id,
                'customer_id' => $customer->id,
                'user_id' => Auth::id(),
                'amount' => -$reward->points_required,
                'type' => 'redeemed',
                'source' => 'reward_redemption',
                'description' => 'استخدام مكافأة: ' . $reward->name,
                'reference_id' => $redemption->id,
                'reference_type' => 'reward_redemption',
            ]);
            $loyaltyPoint->save();

            // زيادة عدد مرات استخدام المكافأة
            $reward->increment('usage_count');

            DB::commit();

            return redirect()->route('reward-redemptions.index')
                ->with('success', 'تم استخدام المكافأة بنجاح');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->with('error', 'حدث خطأ أثناء استخدام المكافأة: ' . $e->getMessage());
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $redemption = RewardRedemption::with(['customer', 'reward', 'user'])
            ->findOrFail($id);

        // التحقق من أن المستخدم ينتمي إلى نفس الشركة
        if ($redemption->reward->company_id != Auth::user()->company_id) {
            abort(403, 'غير مصرح لك بعرض هذا الاستخدام');
        }

        return view('reward_redemptions.show', compact('redemption'));
    }

    /**
     * عرض استخدامات المكافآت لعميل معين
     *
     * @param  int  $customerId
     * @return \Illuminate\Http\Response
     */
    public function customerRedemptions($customerId)
    {
        $customer = Customer::findOrFail($customerId);

        // التحقق من أن المستخدم ينتمي إلى نفس الشركة
        if ($customer->company_id != Auth::user()->company_id) {
            abort(403, 'غير مصرح لك بعرض استخدامات مكافآت هذا العميل');
        }

        $redemptions = RewardRedemption::with(['reward', 'user'])
            ->where('customer_id', $customerId)
            ->orderBy('created_at', 'desc')
            ->paginate(15);

        return view('reward_redemptions.customer_redemptions', compact('redemptions', 'customer'));
    }

    /**
     * عرض استخدامات مكافأة معينة
     *
     * @param  int  $rewardId
     * @return \Illuminate\Http\Response
     */
    public function rewardRedemptions($rewardId)
    {
        $reward = CustomerReward::findOrFail($rewardId);

        // التحقق من أن المستخدم ينتمي إلى نفس الشركة
        if ($reward->company_id != Auth::user()->company_id) {
            abort(403, 'غير مصرح لك بعرض استخدامات هذه المكافأة');
        }

        $redemptions = RewardRedemption::with(['customer', 'user'])
            ->where('reward_id', $rewardId)
            ->orderBy('created_at', 'desc')
            ->paginate(15);

        return view('reward_redemptions.reward_redemptions', compact('redemptions', 'reward'));
    }
}
