<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Invoice;
use App\Models\Payment;
use App\Models\MaintenanceContract;
use App\Models\MaintenanceLog;
use App\Models\ElevatorPart;
use App\Models\InventoryTransaction;
use App\Models\Customer;
use App\Models\Elevator;
use Carbon\Carbon;

class ReportController extends Controller
{
    /**
     * عرض الصفحة الرئيسية للتقارير
     */
    public function index()
    {
        // الحصول على شركة المستخدم الحالي
        $userCompanyId = auth()->user()->company_id;

        // تطبيق عزل الشركات على جميع المستخدمين بما فيهم المشرفين
        // إحصائيات سريعة للصفحة الرئيسية

        // إجمالي الفواتير
        $invoicesQuery = Invoice::query();
        if ($userCompanyId) {
            $invoicesQuery->where('company_id', $userCompanyId);
        }
        $totalInvoices = $invoicesQuery->count();

        // إجمالي المدفوعات
        $paymentsQuery = Payment::query();
        if ($userCompanyId) {
            $paymentsQuery->where('company_id', $userCompanyId);
        }
        $totalPayments = $paymentsQuery->sum('amount');

        // عقود الصيانة النشطة
        $contractsQuery = MaintenanceContract::where('status', 'active');
        if ($userCompanyId) {
            $contractsQuery->where('company_id', $userCompanyId);
        }
        $activeContracts = $contractsQuery->count();

        // المستحقات المتأخرة
        $overdueQuery = Invoice::where('due_date', '<', Carbon::now())
            ->where('status', '!=', 'paid');
        if ($userCompanyId) {
            $overdueQuery->where('company_id', $userCompanyId);
        }
        $overdueAmount = $overdueQuery->sum(\DB::raw('total_amount - paid_amount'));

        return view('reports.index', compact(
            'totalInvoices',
            'totalPayments',
            'activeContracts',
            'overdueAmount'
        ));
    }

    /**
     * عرض تقارير المبيعات
     */
    public function sales(Request $request)
    {
        $startDate = $request->input('start_date', Carbon::now()->startOfMonth()->format('Y-m-d'));
        $endDate = $request->input('end_date', Carbon::now()->format('Y-m-d'));

        // الحصول على شركة المستخدم الحالي
        $userCompanyId = auth()->user()->company_id;

        // تطبيق عزل الشركات على جميع المستخدمين بما فيهم المشرفين
        // الحصول على الفواتير في الفترة المحددة مع تطبيق عزل الشركات
        $invoicesQuery = Invoice::whereBetween('invoice_date', [$startDate, $endDate]);

        // تقييد النتائج بشركة المستخدم لجميع المستخدمين
        if ($userCompanyId) {
            $invoicesQuery->where('company_id', $userCompanyId);
        }

        $invoices = $invoicesQuery->orderBy('invoice_date', 'desc')->get();

        // حساب إجمالي المبيعات
        $totalSales = $invoices->sum('total_amount');

        // حساب إجمالي المدفوعات مع تطبيق عزل الشركات
        $paymentsQuery = Payment::whereBetween('payment_date', [$startDate, $endDate]);

        // تقييد النتائج بشركة المستخدم لجميع المستخدمين
        if ($userCompanyId) {
            $paymentsQuery->where('company_id', $userCompanyId);
        }

        $totalPayments = $paymentsQuery->sum('amount');

        // حساب المبيعات حسب العميل
        $salesByCustomer = $invoices->groupBy('customer_id')
            ->map(function ($customerInvoices) {
                $customer = Customer::find($customerInvoices->first()->customer_id);
                return [
                    'customer_name' => $customer ? $customer->name : 'غير معروف',
                    'total' => $customerInvoices->sum('total_amount')
                ];
            })
            ->sortByDesc('total');

        return view('reports.sales', compact(
            'invoices',
            'totalSales',
            'totalPayments',
            'salesByCustomer',
            'startDate',
            'endDate'
        ));
    }

    /**
     * عرض تقارير الصيانة
     */
    public function maintenance(Request $request)
    {
        $startDate = $request->input('start_date', Carbon::now()->startOfMonth()->format('Y-m-d'));
        $endDate = $request->input('end_date', Carbon::now()->format('Y-m-d'));

        // الحصول على شركة المستخدم الحالي
        $userCompanyId = auth()->user()->company_id;

        // تطبيق عزل الشركات على جميع المستخدمين بما فيهم المشرفين
        // الحصول على عقود الصيانة النشطة مع تطبيق عزل الشركات
        $activeContractsQuery = MaintenanceContract::where('status', 'active');

        // تقييد النتائج بشركة المستخدم لجميع المستخدمين
        if ($userCompanyId) {
            $activeContractsQuery->where('company_id', $userCompanyId);
        }

        $activeContracts = $activeContractsQuery->count();

        // الحصول على سجلات الصيانة في الفترة المحددة مع تطبيق عزل الشركات
        $maintenanceLogsQuery = MaintenanceLog::whereBetween('maintenance_date', [$startDate, $endDate]);

        // تقييد النتائج بشركة المستخدم لجميع المستخدمين
        if ($userCompanyId) {
            $maintenanceLogsQuery->where('company_id', $userCompanyId);
        }

        $maintenanceLogs = $maintenanceLogsQuery->orderBy('maintenance_date', 'desc')->get();

        // حساب عدد عمليات الصيانة حسب النوع
        $maintenanceByType = $maintenanceLogs->groupBy('maintenance_type')
            ->map(function ($logs, $type) {
                return [
                    'type' => $type,
                    'count' => $logs->count()
                ];
            });

        // حساب عدد المصاعد التي تمت صيانتها
        $elevatorsMaintained = $maintenanceLogs->pluck('elevator_id')->unique()->count();

        return view('reports.maintenance', compact(
            'activeContracts',
            'maintenanceLogs',
            'maintenanceByType',
            'elevatorsMaintained',
            'startDate',
            'endDate'
        ));
    }

    /**
     * عرض تقارير المخزون
     */
    public function inventory(Request $request)
    {
        // الحصول على شركة المستخدم الحالي
        $userCompanyId = auth()->user()->company_id;

        // تطبيق عزل الشركات على جميع المستخدمين بما فيهم المشرفين
        // الحصول على قطع الغيار منخفضة المخزون مع تطبيق عزل الشركات
        $lowStockPartsQuery = ElevatorPart::whereRaw('current_stock <= min_stock_level');

        // تقييد النتائج بشركة المستخدم لجميع المستخدمين
        if ($userCompanyId) {
            $lowStockPartsQuery->where('company_id', $userCompanyId);
        }

        $lowStockParts = $lowStockPartsQuery->orderBy('current_stock')->get();

        // الحصول على إجمالي قيمة المخزون مع تطبيق عزل الشركات
        $totalInventoryValueQuery = ElevatorPart::selectRaw('SUM(current_stock * cost_price) as total_value');

        // تقييد النتائج بشركة المستخدم لجميع المستخدمين
        if ($userCompanyId) {
            $totalInventoryValueQuery->where('company_id', $userCompanyId);
        }

        $totalInventoryValue = $totalInventoryValueQuery->first()->total_value ?? 0;

        // الحصول على أكثر قطع الغيار استخدامًا مع تطبيق عزل الشركات
        $mostUsedPartsQuery = InventoryTransaction::where('transaction_type', 'out');

        // تقييد النتائج بشركة المستخدم لجميع المستخدمين
        if ($userCompanyId) {
            $mostUsedPartsQuery->where('company_id', $userCompanyId);
        }

        $mostUsedParts = $mostUsedPartsQuery
            ->selectRaw('item_id, SUM(quantity) as total_used')
            ->groupBy('item_id')
            ->orderByDesc('total_used')
            ->limit(10)
            ->get()
            ->map(function ($transaction) {
                $item = InventoryItem::find($transaction->item_id);
                return [
                    'part_name' => $item ? $item->name : 'غير معروف',
                    'total_used' => $transaction->total_used
                ];
            });

        return view('reports.inventory', compact(
            'lowStockParts',
            'totalInventoryValue',
            'mostUsedParts'
        ));
    }

    /**
     * عرض التقارير المالية
     */
    public function financial(Request $request)
    {
        $startDate = $request->input('start_date', Carbon::now()->startOfMonth()->format('Y-m-d'));
        $endDate = $request->input('end_date', Carbon::now()->format('Y-m-d'));

        // الحصول على شركة المستخدم الحالي
        $userCompanyId = auth()->user()->company_id;

        // تطبيق عزل الشركات على جميع المستخدمين بما فيهم المشرفين
        // الحصول على إجمالي الإيرادات مع تطبيق عزل الشركات
        $totalRevenueQuery = Invoice::whereBetween('invoice_date', [$startDate, $endDate]);

        // تقييد النتائج بشركة المستخدم لجميع المستخدمين
        if ($userCompanyId) {
            $totalRevenueQuery->where('company_id', $userCompanyId);
        }

        $totalRevenue = $totalRevenueQuery->sum('total_amount');

        // الحصول على إجمالي المدفوعات المستلمة مع تطبيق عزل الشركات
        $totalPaymentsReceivedQuery = Payment::whereBetween('payment_date', [$startDate, $endDate]);

        // تقييد النتائج بشركة المستخدم لجميع المستخدمين
        if ($userCompanyId) {
            $totalPaymentsReceivedQuery->where('company_id', $userCompanyId);
        }

        $totalPaymentsReceived = $totalPaymentsReceivedQuery->sum('amount');

        // حساب المستحقات المتأخرة مع تطبيق عزل الشركات
        $overdueAmountQuery = Invoice::where('due_date', '<', Carbon::now())
            ->where('status', '!=', 'paid');

        // تقييد النتائج بشركة المستخدم لجميع المستخدمين
        if ($userCompanyId) {
            $overdueAmountQuery->where('company_id', $userCompanyId);
        }

        $overdueAmount = $overdueAmountQuery->sum(\DB::raw('total_amount - paid_amount'));

        // حساب الإيرادات حسب الشهر مع تطبيق عزل الشركات
        $revenueByMonthQuery = Invoice::whereBetween('invoice_date', [$startDate, $endDate]);

        // تقييد النتائج بشركة المستخدم لجميع المستخدمين
        if ($userCompanyId) {
            $revenueByMonthQuery->where('company_id', $userCompanyId);
        }

        $revenueByMonth = $revenueByMonthQuery
            ->selectRaw('MONTH(invoice_date) as month, YEAR(invoice_date) as year, SUM(total_amount) as total')
            ->groupBy('year', 'month')
            ->orderBy('year')
            ->orderBy('month')
            ->get()
            ->map(function ($item) {
                $monthName = Carbon::createFromDate($item->year, $item->month, 1)->translatedFormat('F Y');
                return [
                    'month' => $monthName,
                    'total' => $item->total
                ];
            });

        return view('reports.financial', compact(
            'totalRevenue',
            'totalPaymentsReceived',
            'overdueAmount',
            'revenueByMonth',
            'startDate',
            'endDate'
        ));
    }
}
