<?php

namespace App\Http\Controllers;

use App\Models\PurchaseReturn;
use App\Models\PurchaseReturnItem;
use App\Models\PurchaseInvoice;
use App\Models\Supplier;
use App\Models\Warehouse;
use App\Models\InventoryItem;
use App\Models\InventoryTransaction;
use App\Services\PurchaseReturnJournalService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;

class PurchaseReturnController extends Controller
{
    protected $journalService;

    public function __construct(PurchaseReturnJournalService $journalService)
    {
        $this->journalService = $journalService;
        $this->middleware('auth');
    }

    /**
     * عرض قائمة مرتجعات المشتريات
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $purchaseReturns = PurchaseReturn::with(['supplier', 'warehouse', 'purchaseInvoice'])
            ->where('company_id', Auth::user()->company_id)
            ->orderBy('created_at', 'desc')
            ->paginate(15);

        return view('inventory.purchase_returns.index', compact('purchaseReturns'));
    }

    /**
     * عرض نموذج إنشاء مرتجع مشتريات جديد
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $companyId = Auth::user()->company_id;

        $suppliers = Supplier::where('company_id', $companyId)
            ->where('is_active', true)
            ->get();

        $warehouses = Warehouse::where('company_id', $companyId)
            ->where('is_active', true)
            ->get();

        $purchaseInvoices = PurchaseInvoice::where('company_id', $companyId)
            ->orderBy('created_at', 'desc')
            ->get();

        $inventoryItems = InventoryItem::where('company_id', $companyId)
            ->where('is_active', true)
            ->get();

        return view('inventory.purchase_returns.create', compact(
            'suppliers',
            'warehouses',
            'purchaseInvoices',
            'inventoryItems'
        ));
    }

    /**
     * تخزين مرتجع مشتريات جديد
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $request->validate([
            'supplier_id' => 'required|exists:suppliers,id',
            'warehouse_id' => 'required|exists:warehouses,id',
            'purchase_invoice_id' => 'nullable|exists:purchase_invoices,id',
            'return_date' => 'required|date',
            'items' => 'required|array|min:1',
            'items.*.inventory_item_id' => 'required|exists:inventory_items,id',
            'items.*.quantity' => 'required|numeric|min:0.01',
            'items.*.unit_price' => 'required|numeric|min:0',
            'items.*.tax_rate' => 'nullable|numeric|min:0',
            'items.*.discount_amount' => 'nullable|numeric|min:0',
        ]);

        try {
            DB::beginTransaction();

            // إنشاء مرتجع المشتريات
            $purchaseReturn = new PurchaseReturn();
            $purchaseReturn->return_number = PurchaseReturn::generateReturnNumber(Auth::user()->company_id);
            $purchaseReturn->company_id = Auth::user()->company_id;
            $purchaseReturn->supplier_id = $request->supplier_id;
            $purchaseReturn->warehouse_id = $request->warehouse_id;
            $purchaseReturn->purchase_invoice_id = $request->purchase_invoice_id;
            $purchaseReturn->created_by = Auth::id();
            $purchaseReturn->return_date = $request->return_date;
            $purchaseReturn->notes = $request->notes;

            // حساب المجاميع
            $subtotal = 0;
            $taxAmount = 0;
            $discountAmount = 0;

            foreach ($request->items as $item) {
                $itemSubtotal = $item['quantity'] * $item['unit_price'];
                $itemTax = isset($item['tax_rate']) ? ($itemSubtotal * $item['tax_rate'] / 100) : 0;
                $itemDiscount = isset($item['discount_amount']) ? $item['discount_amount'] : 0;

                $subtotal += $itemSubtotal;
                $taxAmount += $itemTax;
                $discountAmount += $itemDiscount;
            }

            $purchaseReturn->subtotal = $subtotal;
            $purchaseReturn->tax_amount = $taxAmount;
            $purchaseReturn->discount_amount = $discountAmount;
            $purchaseReturn->total_amount = $subtotal + $taxAmount - $discountAmount;

            // حفظ مرفق إذا وجد
            if ($request->hasFile('attachment')) {
                $file = $request->file('attachment');
                $fileName = time() . '_' . $file->getClientOriginalName();
                $file->move(public_path('uploads/purchase_returns'), $fileName);
                $purchaseReturn->attachment = 'uploads/purchase_returns/' . $fileName;
            }

            $purchaseReturn->save();

            // إضافة عناصر المرتجع
            foreach ($request->items as $item) {
                $returnItem = new PurchaseReturnItem();
                $returnItem->purchase_return_id = $purchaseReturn->id;
                $returnItem->company_id = Auth::user()->company_id;
                $returnItem->inventory_item_id = $item['inventory_item_id'];
                $returnItem->purchase_invoice_item_id = $item['purchase_invoice_item_id'] ?? null;
                $returnItem->quantity = $item['quantity'];
                $returnItem->unit_price = $item['unit_price'];
                $returnItem->tax_rate = $item['tax_rate'] ?? 0;
                $returnItem->tax_amount = ($item['quantity'] * $item['unit_price']) * ($item['tax_rate'] ?? 0) / 100;
                $returnItem->discount_amount = $item['discount_amount'] ?? 0;
                $returnItem->total_amount = ($item['quantity'] * $item['unit_price']) + $returnItem->tax_amount - $returnItem->discount_amount;
                $returnItem->notes = $item['notes'] ?? null;
                $returnItem->save();

                // تحديث المخزون - إنقاص الكمية المرتجعة
                $inventoryTransaction = new InventoryTransaction();
                $inventoryTransaction->company_id = Auth::user()->company_id;
                $inventoryTransaction->warehouse_id = $purchaseReturn->warehouse_id;
                $inventoryTransaction->item_id = $item['inventory_item_id'];
                $inventoryTransaction->transaction_type = 'return_to_supplier';
                $inventoryTransaction->reference_id = $purchaseReturn->id;
                $inventoryTransaction->reference_type = PurchaseReturn::class;
                $inventoryTransaction->quantity = -1 * $item['quantity']; // سالب لأنه خروج من المخزون
                $inventoryTransaction->unit_price = $item['unit_price'];
                $inventoryTransaction->total_price = $item['quantity'] * $item['unit_price'];
                $inventoryTransaction->transaction_date = $purchaseReturn->return_date;
                $inventoryTransaction->created_by = Auth::id();
                $inventoryTransaction->notes = "مرتجع مشتريات رقم: " . $purchaseReturn->return_number;
                $inventoryTransaction->save();
            }

            // إنشاء القيد المحاسبي
            $this->journalService->createPurchaseReturnJournalEntry($purchaseReturn);

            DB::commit();

            return redirect()->route('inventory.purchase-returns.show', $purchaseReturn->id)
                ->with('success', 'تم إنشاء مرتجع المشتريات بنجاح');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()
                ->withInput()
                ->with('error', 'حدث خطأ أثناء إنشاء مرتجع المشتريات: ' . $e->getMessage());
        }
    }

    /**
     * عرض تفاصيل مرتجع مشتريات
     *
     * @param  \App\Models\PurchaseReturn  $purchaseReturn
     * @return \Illuminate\Http\Response
     */
    public function show(PurchaseReturn $purchaseReturn)
    {
        // التحقق من أن المرتجع ينتمي للشركة الحالية
        if ($purchaseReturn->company_id != Auth::user()->company_id) {
            return redirect()->route('inventory.purchase-returns.index')
                ->with('error', 'غير مصرح لك بالوصول إلى هذا المرتجع');
        }

        $purchaseReturn->load(['supplier', 'warehouse', 'purchaseInvoice', 'items.inventoryItem', 'journalEntry']);

        return view('inventory.purchase_returns.show', compact('purchaseReturn'));
    }

    /**
     * عرض نموذج تعديل مرتجع مشتريات
     *
     * @param  \App\Models\PurchaseReturn  $purchaseReturn
     * @return \Illuminate\Http\Response
     */
    public function edit(PurchaseReturn $purchaseReturn)
    {
        // التحقق من أن المرتجع ينتمي للشركة الحالية
        if ($purchaseReturn->company_id != Auth::user()->company_id) {
            return redirect()->route('inventory.purchase-returns.index')
                ->with('error', 'غير مصرح لك بالوصول إلى هذا المرتجع');
        }

        $companyId = Auth::user()->company_id;
        $purchaseReturn->load(['items.inventoryItem']);

        $suppliers = Supplier::where('company_id', $companyId)
            ->where('is_active', true)
            ->get();

        $warehouses = Warehouse::where('company_id', $companyId)
            ->where('is_active', true)
            ->get();

        $purchaseInvoices = PurchaseInvoice::where('company_id', $companyId)
            ->orderBy('created_at', 'desc')
            ->get();

        $inventoryItems = InventoryItem::where('company_id', $companyId)
            ->where('is_active', true)
            ->get();

        return view('inventory.purchase_returns.edit', compact(
            'purchaseReturn',
            'suppliers',
            'warehouses',
            'purchaseInvoices',
            'inventoryItems'
        ));
    }

    /**
     * تحديث مرتجع مشتريات
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\PurchaseReturn  $purchaseReturn
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, PurchaseReturn $purchaseReturn)
    {
        // التحقق من أن المرتجع ينتمي للشركة الحالية
        if ($purchaseReturn->company_id != Auth::user()->company_id) {
            return redirect()->route('inventory.purchase-returns.index')
                ->with('error', 'غير مصرح لك بالوصول إلى هذا المرتجع');
        }
        $request->validate([
            'supplier_id' => 'required|exists:suppliers,id',
            'warehouse_id' => 'required|exists:warehouses,id',
            'purchase_invoice_id' => 'nullable|exists:purchase_invoices,id',
            'return_date' => 'required|date',
            'items' => 'required|array|min:1',
            'items.*.inventory_item_id' => 'required|exists:inventory_items,id',
            'items.*.quantity' => 'required|numeric|min:0.01',
            'items.*.unit_price' => 'required|numeric|min:0',
            'items.*.tax_rate' => 'nullable|numeric|min:0',
            'items.*.discount_amount' => 'nullable|numeric|min:0',
        ]);

        try {
            DB::beginTransaction();

            // تحديث بيانات المرتجع
            $purchaseReturn->supplier_id = $request->supplier_id;
            $purchaseReturn->warehouse_id = $request->warehouse_id;
            $purchaseReturn->purchase_invoice_id = $request->purchase_invoice_id;
            $purchaseReturn->return_date = $request->return_date;
            $purchaseReturn->notes = $request->notes;

            // حساب المجاميع
            $subtotal = 0;
            $taxAmount = 0;
            $discountAmount = 0;

            // حذف العناصر القديمة وإلغاء معاملات المخزون المرتبطة بها
            foreach ($purchaseReturn->items as $item) {
                // إلغاء معاملة المخزون السابقة
                InventoryTransaction::where('reference_id', $purchaseReturn->id)
                    ->where('reference_type', PurchaseReturn::class)
                    ->where('item_id', $item->inventory_item_id)
                    ->delete();

                // حذف عنصر المرتجع
                $item->delete();
            }

            // إضافة العناصر الجديدة
            foreach ($request->items as $item) {
                $itemSubtotal = $item['quantity'] * $item['unit_price'];
                $itemTax = isset($item['tax_rate']) ? ($itemSubtotal * $item['tax_rate'] / 100) : 0;
                $itemDiscount = isset($item['discount_amount']) ? $item['discount_amount'] : 0;

                $subtotal += $itemSubtotal;
                $taxAmount += $itemTax;
                $discountAmount += $itemDiscount;

                $returnItem = new PurchaseReturnItem();
                $returnItem->purchase_return_id = $purchaseReturn->id;
                $returnItem->company_id = Auth::user()->company_id;
                $returnItem->inventory_item_id = $item['inventory_item_id'];
                $returnItem->purchase_invoice_item_id = $item['purchase_invoice_item_id'] ?? null;
                $returnItem->quantity = $item['quantity'];
                $returnItem->unit_price = $item['unit_price'];
                $returnItem->tax_rate = $item['tax_rate'] ?? 0;
                $returnItem->tax_amount = $itemTax;
                $returnItem->discount_amount = $itemDiscount;
                $returnItem->total_amount = $itemSubtotal + $itemTax - $itemDiscount;
                $returnItem->notes = $item['notes'] ?? null;
                $returnItem->save();

                // إنشاء معاملة مخزون جديدة
                $inventoryTransaction = new InventoryTransaction();
                $inventoryTransaction->company_id = Auth::user()->company_id;
                $inventoryTransaction->warehouse_id = $purchaseReturn->warehouse_id;
                $inventoryTransaction->item_id = $item['inventory_item_id'];
                $inventoryTransaction->transaction_type = 'return_to_supplier';
                $inventoryTransaction->reference_id = $purchaseReturn->id;
                $inventoryTransaction->reference_type = PurchaseReturn::class;
                $inventoryTransaction->quantity = -1 * $item['quantity']; // سالب لأنه خروج من المخزون
                $inventoryTransaction->unit_price = $item['unit_price'];
                $inventoryTransaction->total_price = $item['quantity'] * $item['unit_price'];
                $inventoryTransaction->transaction_date = $purchaseReturn->return_date;
                $inventoryTransaction->created_by = Auth::id();
                $inventoryTransaction->notes = "مرتجع مشتريات رقم: " . $purchaseReturn->return_number;
                $inventoryTransaction->save();
            }

            // تحديث المجاميع
            $purchaseReturn->subtotal = $subtotal;
            $purchaseReturn->tax_amount = $taxAmount;
            $purchaseReturn->discount_amount = $discountAmount;
            $purchaseReturn->total_amount = $subtotal + $taxAmount - $discountAmount;

            // تحديث المرفق إذا وجد
            if ($request->hasFile('attachment')) {
                // حذف المرفق القديم إذا وجد
                if ($purchaseReturn->attachment && file_exists(public_path($purchaseReturn->attachment))) {
                    unlink(public_path($purchaseReturn->attachment));
                }

                $file = $request->file('attachment');
                $fileName = time() . '_' . $file->getClientOriginalName();
                $file->move(public_path('uploads/purchase_returns'), $fileName);
                $purchaseReturn->attachment = 'uploads/purchase_returns/' . $fileName;
            }

            $purchaseReturn->save();

            // تحديث القيد المحاسبي
            if ($purchaseReturn->journalEntry) {
                $this->journalService->updatePurchaseReturnJournalEntry($purchaseReturn);
            } else {
                $this->journalService->createPurchaseReturnJournalEntry($purchaseReturn);
            }

            DB::commit();

            return redirect()->route('inventory.purchase-returns.show', $purchaseReturn->id)
                ->with('success', 'تم تحديث مرتجع المشتريات بنجاح');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()
                ->withInput()
                ->with('error', 'حدث خطأ أثناء تحديث مرتجع المشتريات: ' . $e->getMessage());
        }
    }

    /**
     * حذف مرتجع مشتريات
     *
     * @param  \App\Models\PurchaseReturn  $purchaseReturn
     * @return \Illuminate\Http\Response
     */
    public function destroy(PurchaseReturn $purchaseReturn)
    {
        // التحقق من أن المرتجع ينتمي للشركة الحالية
        if ($purchaseReturn->company_id != Auth::user()->company_id) {
            return redirect()->route('inventory.purchase-returns.index')
                ->with('error', 'غير مصرح لك بالوصول إلى هذا المرتجع');
        }
        try {
            DB::beginTransaction();

            // حذف معاملات المخزون المرتبطة
            InventoryTransaction::where('reference_id', $purchaseReturn->id)
                ->where('reference_type', PurchaseReturn::class)
                ->delete();

            // حذف القيد المحاسبي المرتبط
            if ($purchaseReturn->journalEntry) {
                $purchaseReturn->journalEntry->items()->delete();
                $purchaseReturn->journalEntry->delete();
            }

            // حذف عناصر المرتجع
            $purchaseReturn->items()->delete();

            // حذف المرفق إذا وجد
            if ($purchaseReturn->attachment && file_exists(public_path($purchaseReturn->attachment))) {
                unlink(public_path($purchaseReturn->attachment));
            }

            // حذف المرتجع
            $purchaseReturn->delete();

            DB::commit();

            return redirect()->route('inventory.purchase-returns.index')
                ->with('success', 'تم حذف مرتجع المشتريات بنجاح');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()
                ->with('error', 'حدث خطأ أثناء حذف مرتجع المشتريات: ' . $e->getMessage());
        }
    }

    /**
     * طباعة مرتجع المشتريات
     *
     * @param  \App\Models\PurchaseReturn  $purchaseReturn
     * @return \Illuminate\Http\Response
     */
    public function print(PurchaseReturn $purchaseReturn)
    {
        // التحقق من أن المرتجع ينتمي للشركة الحالية
        if ($purchaseReturn->company_id != Auth::user()->company_id) {
            return redirect()->route('inventory.purchase-returns.index')
                ->with('error', 'غير مصرح لك بالوصول إلى هذا المرتجع');
        }
        $purchaseReturn->load(['supplier', 'warehouse', 'purchaseInvoice', 'items.inventoryItem', 'creator']);

        return view('inventory.purchase_returns.print', compact('purchaseReturn'));
    }

    /**
     * إنشاء مرتجع مشتريات من فاتورة مشتريات
     *
     * @param  \App\Models\PurchaseInvoice  $purchaseInvoice
     * @return \Illuminate\Http\Response
     */
    public function createFromInvoice(PurchaseInvoice $purchaseInvoice)
    {
        // التحقق من أن الفاتورة تنتمي للشركة الحالية
        if ($purchaseInvoice->company_id != Auth::user()->company_id) {
            return redirect()->route('inventory.purchase-invoices.index')
                ->with('error', 'غير مصرح لك بالوصول إلى هذه الفاتورة');
        }

        $companyId = Auth::user()->company_id;
        $purchaseInvoice->load(['items.inventoryItem']);

        $suppliers = Supplier::where('company_id', $companyId)
            ->where('is_active', true)
            ->get();

        $warehouses = Warehouse::where('company_id', $companyId)
            ->where('is_active', true)
            ->get();

        $inventoryItems = InventoryItem::where('company_id', $companyId)
            ->where('is_active', true)
            ->get();

        return view('inventory.purchase_returns.create_from_invoice', compact(
            'purchaseInvoice',
            'suppliers',
            'warehouses',
            'inventoryItems'
        ));
    }
}
