<?php

namespace App\Http\Controllers;

use App\Models\PurchaseInvoice;
use App\Models\PurchaseInvoiceItem;
use App\Models\PurchaseOrder;
use App\Models\Supplier;
use App\Models\Warehouse;
use App\Models\InventoryItem;
use App\Models\InventoryStock;
use App\Models\InventoryTransaction;
use App\Services\PurchaseInvoiceJournalService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;

class PurchaseInvoiceController extends Controller
{
    /**
     * تحديث المخزون بناءً على فاتورة المشتريات
     *
     * @param PurchaseInvoice $invoice
     * @return void
     */
    private function updateInventory(PurchaseInvoice $invoice)
    {
        $companyId = $invoice->company_id;
        $warehouseId = $invoice->warehouse_id;

        foreach ($invoice->items as $invoiceItem) {
            $itemId = $invoiceItem->item_id;
            $quantity = $invoiceItem->quantity;

            // الحصول على سجل المخزون أو إنشاء واحد جديد
            $stock = InventoryStock::firstOrNew([
                'company_id' => $companyId,
                'warehouse_id' => $warehouseId,
                'item_id' => $itemId,
            ]);

            if (!$stock->exists) {
                $stock->quantity = 0;
                $stock->reserved_quantity = 0;
            }

            // زيادة كمية المخزون
            $stock->quantity += $quantity;
            $stock->save();

            // إنشاء معاملة مخزون
            $transaction = new InventoryTransaction();
            $transaction->company_id = $companyId;
            $transaction->transaction_type = 'purchase';
            $transaction->item_id = $itemId;
            $transaction->warehouse_id = $warehouseId;
            $transaction->quantity = $quantity;
            $transaction->unit_price = $invoiceItem->unit_price;
            $transaction->reference_number = $invoice->invoice_number;
            $transaction->reference_type = 'purchase_invoice';
            $transaction->reference_id = $invoice->id;
            $transaction->created_by = auth()->id();
            $transaction->notes = 'تم إضافة المخزون من فاتورة المشتريات #' . $invoice->invoice_number;
            $transaction->save();
        }
    }

    /**
     * تحديث المخزون عند تعديل فاتورة المشتريات
     *
     * @param PurchaseInvoice $invoice
     * @param array $oldItems
     * @return void
     */
    private function updateInventoryOnEdit(PurchaseInvoice $invoice, array $oldItems)
    {
        $companyId = $invoice->company_id;
        $warehouseId = $invoice->warehouse_id;

        // تجميع العناصر القديمة والجديدة للمقارنة
        $newItems = $invoice->items->keyBy('id');

        // معالجة العناصر المحدثة والجديدة
        foreach ($invoice->items as $invoiceItem) {
            $itemId = $invoiceItem->item_id;
            $newQuantity = $invoiceItem->quantity;

            // التحقق مما إذا كان هذا عنصر موجود مسبقًا أو جديد
            if (isset($oldItems[$invoiceItem->id])) {
                // عنصر موجود مسبقًا - حساب الفرق في الكمية
                $oldQuantity = $oldItems[$invoiceItem->id]['quantity'];
                $quantityDifference = $newQuantity - $oldQuantity;

                // إذا لم يكن هناك تغيير في الكمية، تخطي هذا العنصر
                if ($quantityDifference == 0) {
                    continue;
                }

                // الحصول على سجل المخزون
                $stock = InventoryStock::firstOrNew([
                    'company_id' => $companyId,
                    'warehouse_id' => $warehouseId,
                    'item_id' => $itemId,
                ]);

                if (!$stock->exists) {
                    $stock->quantity = 0;
                    $stock->reserved_quantity = 0;
                }

                // تحديث كمية المخزون بناءً على الفرق
                $stock->quantity += $quantityDifference;
                $stock->save();

                // إنشاء معاملة مخزون للتعديل
                $transaction = new InventoryTransaction();
                $transaction->company_id = $companyId;
                $transaction->transaction_type = 'adjustment';
                $transaction->item_id = $itemId;
                $transaction->warehouse_id = $warehouseId;
                $transaction->quantity = $quantityDifference;
                $transaction->unit_price = $invoiceItem->unit_price;
                $transaction->reference_number = $invoice->invoice_number;
                $transaction->reference_type = 'purchase_invoice_edit';
                $transaction->reference_id = $invoice->id;
                $transaction->created_by = auth()->id();
                $transaction->notes = 'تعديل كمية المخزون من فاتورة المشتريات #' . $invoice->invoice_number;
                $transaction->save();
            } else {
                // عنصر جديد - إضافة للمخزون
                $stock = InventoryStock::firstOrNew([
                    'company_id' => $companyId,
                    'warehouse_id' => $warehouseId,
                    'item_id' => $itemId,
                ]);

                if (!$stock->exists) {
                    $stock->quantity = 0;
                    $stock->reserved_quantity = 0;
                }

                // زيادة كمية المخزون
                $stock->quantity += $newQuantity;
                $stock->save();

                // إنشاء معاملة مخزون للإضافة
                $transaction = new InventoryTransaction();
                $transaction->company_id = $companyId;
                $transaction->transaction_type = 'purchase';
                $transaction->item_id = $itemId;
                $transaction->warehouse_id = $warehouseId;
                $transaction->quantity = $newQuantity;
                $transaction->unit_price = $invoiceItem->unit_price;
                $transaction->reference_number = $invoice->invoice_number;
                $transaction->reference_type = 'purchase_invoice';
                $transaction->reference_id = $invoice->id;
                $transaction->created_by = auth()->id();
                $transaction->notes = 'إضافة عنصر جديد للمخزون من فاتورة المشتريات #' . $invoice->invoice_number;
                $transaction->save();
            }
        }

        // معالجة العناصر المحذوفة
        foreach ($oldItems as $oldItemId => $oldItem) {
            if (!$newItems->has($oldItemId)) {
                // عنصر محذوف - تقليل المخزون
                $itemId = $oldItem['item_id'];
                $oldQuantity = $oldItem['quantity'];

                $stock = InventoryStock::where([
                    'company_id' => $companyId,
                    'warehouse_id' => $warehouseId,
                    'item_id' => $itemId,
                ])->first();

                if ($stock) {
                    // تقليل كمية المخزون
                    $stock->quantity -= $oldQuantity;
                    $stock->save();

                    // إنشاء معاملة مخزون للحذف
                    $transaction = new InventoryTransaction();
                    $transaction->company_id = $companyId;
                    $transaction->transaction_type = 'adjustment';
                    $transaction->item_id = $itemId;
                    $transaction->warehouse_id = $warehouseId;
                    $transaction->quantity = -$oldQuantity;
                    $transaction->unit_price = $oldItem['unit_price'];
                    $transaction->reference_number = $invoice->invoice_number;
                    $transaction->reference_type = 'purchase_invoice_edit';
                    $transaction->reference_id = $invoice->id;
                    $transaction->created_by = auth()->id();
                    $transaction->notes = 'حذف عنصر من المخزون من فاتورة المشتريات #' . $invoice->invoice_number;
                    $transaction->save();
                }
            }
        }
    }
    /**
     * عرض قائمة فواتير المشتريات
     */
    public function index()
    {
        $this->authorize('view_purchase_invoices');

        // الحصول على الشركة الحالية للمستخدم
        $companyId = auth()->user()->company_id;

        // تطبيق عزل الشركات على الفواتير
        $invoices = PurchaseInvoice::with(['supplier', 'warehouse', 'creator'])
            ->where('company_id', $companyId)
            ->orderBy('created_at', 'desc')
            ->paginate(15);

        return view('inventory.purchase-invoices.index', compact('invoices'));
    }

    /**
     * عرض نموذج إنشاء فاتورة مشتريات جديدة
     */
    public function create()
    {
        $this->authorize('create_purchase_invoices');

        // الحصول على الشركة الحالية للمستخدم
        $companyId = auth()->user()->company_id;

        // تطبيق عزل الشركات على البيانات
        $suppliers = Supplier::where('company_id', $companyId)->orderBy('name')->get();
        $warehouses = Warehouse::where('company_id', $companyId)->orderBy('name')->get();
        $items = InventoryItem::where('company_id', $companyId)->orderBy('name')->get();
        $purchaseOrders = PurchaseOrder::where('company_id', $companyId)
            ->where(function ($query) {
                $query->where('status', 'completed')
                    ->orWhere('status', 'partially_received');
            })
            ->orderBy('created_at', 'desc')
            ->get();

        return view('inventory.purchase-invoices.create', compact('suppliers', 'warehouses', 'items', 'purchaseOrders'));
    }

    /**
     * حفظ فاتورة مشتريات جديدة
     */
    public function store(Request $request)
    {
        $this->authorize('create_purchase_invoices');

        // الحصول على الشركة الحالية للمستخدم
        $companyId = auth()->user()->company_id;

        $validated = $request->validate([
            'invoice_number' => [
                'required',
                'string',
                // تحقق من فريد رقم الفاتورة على مستوى الشركة فقط
                function ($attribute, $value, $fail) use ($companyId) {
                    $exists = PurchaseInvoice::where('invoice_number', $value)
                        ->where('company_id', $companyId)
                        ->exists();

                    if ($exists) {
                        $fail('رقم الفاتورة مستخدم بالفعل في هذه الشركة.');
                    }
                }
            ],
            'supplier_id' => 'required|exists:suppliers,id',
            'warehouse_id' => 'required|exists:warehouses,id',
            'purchase_order_id' => 'nullable|exists:purchase_orders,id',
            'invoice_date' => 'required|date',
            'due_date' => 'nullable|date|after_or_equal:invoice_date',
            'subtotal' => 'required|numeric|min:0',
            'tax_amount' => 'required|numeric|min:0',
            'discount_amount' => 'required|numeric|min:0',
            'total_amount' => 'required|numeric|min:0',
            'notes' => 'nullable|string',
            'supplier_invoice_number' => 'nullable|string',
            'attachment' => 'nullable|file|mimes:pdf,jpg,jpeg,png|max:2048',
            'items' => 'required|array|min:1',
            'items.*.item_id' => 'required|exists:inventory_items,id',
            'items.*.purchase_order_item_id' => 'nullable|exists:purchase_order_items,id',
            'items.*.quantity' => 'required|numeric|min:0.01',
            'items.*.unit_price' => 'required|numeric|min:0',
            'items.*.tax_rate' => 'required|numeric|min:0',
            'items.*.tax_amount' => 'required|numeric|min:0',
            'items.*.discount_rate' => 'required|numeric|min:0',
            'items.*.discount_amount' => 'required|numeric|min:0',
            'items.*.total_price' => 'required|numeric|min:0',
            'items.*.notes' => 'nullable|string',
        ]);

        // معالجة الملف المرفق إذا وجد
        $attachmentPath = null;
        if ($request->hasFile('attachment')) {
            $file = $request->file('attachment');
            $fileName = 'invoice_' . time() . '_' . Str::random(10) . '.' . $file->getClientOriginalExtension();
            $attachmentPath = $file->storeAs('purchase_invoices', $fileName, 'public');
        }

        DB::beginTransaction();

        try {
            // إنشاء فاتورة المشتريات
            $invoice = PurchaseInvoice::create([
                'invoice_number' => $validated['invoice_number'],
                'company_id' => auth()->user()->company_id, // إضافة معرف الشركة
                'supplier_id' => $validated['supplier_id'],
                'warehouse_id' => $validated['warehouse_id'],
                'purchase_order_id' => $validated['purchase_order_id'] ?? null,
                'created_by' => auth()->id(),
                'invoice_date' => $validated['invoice_date'],
                'due_date' => $validated['due_date'] ?? null,
                'subtotal' => $validated['subtotal'],
                'tax_amount' => $validated['tax_amount'],
                'discount_amount' => $validated['discount_amount'],
                'total_amount' => $validated['total_amount'],
                'payment_status' => 'unpaid',
                'notes' => $validated['notes'] ?? null,
                'supplier_invoice_number' => $validated['supplier_invoice_number'] ?? null,
                'attachment' => $attachmentPath,
            ]);

            // إضافة عناصر الفاتورة
            foreach ($validated['items'] as $item) {
                PurchaseInvoiceItem::create([
                    'purchase_invoice_id' => $invoice->id,
                    'item_id' => $item['item_id'],
                    'purchase_order_item_id' => $item['purchase_order_item_id'] ?? null,
                    'quantity' => $item['quantity'],
                    'unit_price' => $item['unit_price'],
                    'tax_rate' => $item['tax_rate'],
                    'tax_amount' => $item['tax_amount'],
                    'discount_rate' => $item['discount_rate'],
                    'discount_amount' => $item['discount_amount'],
                    'total_price' => $item['total_price'],
                    'notes' => $item['notes'] ?? null,
                ]);
            }

            // تحديث المخزون بناءً على فاتورة المشتريات
            $this->updateInventory($invoice);

            // إنشاء قيد محاسبي تلقائي لفاتورة المشتريات
            $journalService = new PurchaseInvoiceJournalService();
            $journalEntry = $journalService->createPurchaseInvoiceJournalEntry($invoice);

            DB::commit();

            $successMessage = 'تم إنشاء فاتورة المشتريات وتحديث المخزون بنجاح';
            if ($journalEntry) {
                $successMessage .= ' وتم إنشاء قيد محاسبي تلقائي برقم: ' . $journalEntry->entry_number;
            } else {
                $successMessage .= ' ولكن فشل إنشاء القيد المحاسبي التلقائي';
            }

            return redirect()->route('inventory.purchase-invoices.show', $invoice)
                ->with('success', $successMessage);
        } catch (\Exception $e) {
            DB::rollBack();

            // حذف الملف المرفق في حالة حدوث خطأ
            if ($attachmentPath) {
                Storage::disk('public')->delete($attachmentPath);
            }

            return redirect()->back()
                ->withInput()
                ->with('error', 'حدث خطأ أثناء إنشاء فاتورة المشتريات: ' . $e->getMessage());
        }
    }

    /**
     * عرض تفاصيل فاتورة مشتريات
     */
    public function show(PurchaseInvoice $purchaseInvoice)
    {
        $this->authorize('view_purchase_invoices');

        // التحقق من أن الفاتورة تنتمي لنفس شركة المستخدم
        $companyId = auth()->user()->company_id;

        // تعديل: السماح بعرض الفواتير التي ليس لها company_id أو تنتمي لنفس شركة المستخدم
        if ($purchaseInvoice->company_id !== null && $purchaseInvoice->company_id != $companyId) {
            abort(403, 'غير مصرح لك بعرض فواتير شركات أخرى');
        }

        // إذا لم يكن للفاتورة company_id، قم بتعيينه تلقائياً لشركة المستخدم الحالي
        if ($purchaseInvoice->company_id === null) {
            $purchaseInvoice->update(['company_id' => $companyId]);
        }

        $purchaseInvoice->load(['supplier', 'warehouse', 'creator', 'purchaseOrder', 'items.item', 'payments']);

        return view('inventory.purchase-invoices.show', compact('purchaseInvoice'));
    }

    /**
     * عرض نموذج تعديل فاتورة مشتريات
     */
    public function edit(PurchaseInvoice $purchaseInvoice)
    {
        $this->authorize('edit_purchase_invoices');

        // لا يمكن تعديل الفواتير المدفوعة بالكامل
        if ($purchaseInvoice->payment_status === 'paid') {
            return redirect()->route('inventory.purchase-invoices.show', $purchaseInvoice)
                ->with('error', 'لا يمكن تعديل الفواتير المدفوعة بالكامل');
        }

        // التحقق من أن الفاتورة تنتمي لنفس شركة المستخدم
        $companyId = auth()->user()->company_id;

        // تعديل: السماح بتعديل الفواتير التي ليس لها company_id أو تنتمي لنفس شركة المستخدم
        if ($purchaseInvoice->company_id !== null && $purchaseInvoice->company_id != $companyId) {
            abort(403, 'غير مصرح لك بتعديل فواتير شركات أخرى');
        }

        // إذا لم يكن للفاتورة company_id، قم بتعيينه تلقائياً لشركة المستخدم الحالي
        if ($purchaseInvoice->company_id === null) {
            $purchaseInvoice->update(['company_id' => $companyId]);
        }

        $purchaseInvoice->load(['items.item']);

        // تطبيق عزل الشركات على البيانات
        $suppliers = Supplier::where('company_id', $companyId)->orderBy('name')->get();
        $warehouses = Warehouse::where('company_id', $companyId)->orderBy('name')->get();
        $items = InventoryItem::where('company_id', $companyId)->orderBy('name')->get();
        $purchaseOrders = PurchaseOrder::where('company_id', $companyId)
            ->where(function ($query) {
                $query->where('status', 'completed')
                    ->orWhere('status', 'partially_received');
            })
            ->orderBy('created_at', 'desc')
            ->get();

        return view('inventory.purchase-invoices.edit', compact('purchaseInvoice', 'suppliers', 'warehouses', 'items', 'purchaseOrders'));
    }

    /**
     * تحديث فاتورة مشتريات
     */
    public function update(Request $request, PurchaseInvoice $purchaseInvoice)
    {
        $this->authorize('edit_purchase_invoices');

        // لا يمكن تعديل الفواتير المدفوعة بالكامل
        if ($purchaseInvoice->payment_status === 'paid') {
            return redirect()->route('inventory.purchase-invoices.show', $purchaseInvoice)
                ->with('error', 'لا يمكن تعديل الفواتير المدفوعة بالكامل');
        }

        // الحصول على الشركة الحالية للمستخدم
        $companyId = auth()->user()->company_id;

        $validated = $request->validate([
            'invoice_number' => [
                'required',
                'string',
                // تحقق من فريد رقم الفاتورة على مستوى الشركة فقط
                function ($attribute, $value, $fail) use ($purchaseInvoice, $companyId) {
                    $exists = PurchaseInvoice::where('invoice_number', $value)
                        ->where('company_id', $companyId)
                        ->where('id', '!=', $purchaseInvoice->id)
                        ->exists();

                    if ($exists) {
                        $fail('رقم الفاتورة مستخدم بالفعل في هذه الشركة.');
                    }
                }
            ],
            'supplier_id' => 'required|exists:suppliers,id',
            'warehouse_id' => 'required|exists:warehouses,id',
            'purchase_order_id' => 'nullable|exists:purchase_orders,id',
            'invoice_date' => 'required|date',
            'due_date' => 'nullable|date|after_or_equal:invoice_date',
            'subtotal' => 'required|numeric|min:0',
            'tax_amount' => 'required|numeric|min:0',
            'discount_amount' => 'required|numeric|min:0',
            'total_amount' => 'required|numeric|min:0',
            'notes' => 'nullable|string',
            'supplier_invoice_number' => 'nullable|string',
            'attachment' => 'nullable|file|mimes:pdf,jpg,jpeg,png|max:2048',
            'items' => 'required|array|min:1',
            'items.*.id' => 'nullable|exists:purchase_invoice_items,id',
            'items.*.item_id' => 'required|exists:inventory_items,id',
            'items.*.purchase_order_item_id' => 'nullable|exists:purchase_order_items,id',
            'items.*.quantity' => 'required|numeric|min:0.01',
            'items.*.unit_price' => 'required|numeric|min:0',
            'items.*.tax_rate' => 'required|numeric|min:0',
            'items.*.tax_amount' => 'required|numeric|min:0',
            'items.*.discount_rate' => 'required|numeric|min:0',
            'items.*.discount_amount' => 'required|numeric|min:0',
            'items.*.total_price' => 'required|numeric|min:0',
            'items.*.notes' => 'nullable|string',
        ]);

        // معالجة الملف المرفق إذا وجد
        $attachmentPath = $purchaseInvoice->attachment;
        if ($request->hasFile('attachment')) {
            // حذف الملف القديم إذا وجد
            if ($attachmentPath) {
                Storage::disk('public')->delete($attachmentPath);
            }

            $file = $request->file('attachment');
            $fileName = 'invoice_' . time() . '_' . Str::random(10) . '.' . $file->getClientOriginalExtension();
            $attachmentPath = $file->storeAs('purchase_invoices', $fileName, 'public');
        }

        DB::beginTransaction();

        try {
            // حفظ الكميات القديمة قبل التحديث
            $oldItems = $purchaseInvoice->items->keyBy('id')->toArray();

            // تحديث فاتورة المشتريات
            $purchaseInvoice->update([
                'invoice_number' => $validated['invoice_number'],
                // لا نقوم بتحديث company_id للحفاظ على عزل الشركات
                'supplier_id' => $validated['supplier_id'],
                'warehouse_id' => $validated['warehouse_id'],
                'purchase_order_id' => $validated['purchase_order_id'] ?? null,
                'invoice_date' => $validated['invoice_date'],
                'due_date' => $validated['due_date'] ?? null,
                'subtotal' => $validated['subtotal'],
                'tax_amount' => $validated['tax_amount'],
                'discount_amount' => $validated['discount_amount'],
                'total_amount' => $validated['total_amount'],
                'notes' => $validated['notes'] ?? null,
                'supplier_invoice_number' => $validated['supplier_invoice_number'] ?? null,
                'attachment' => $attachmentPath,
            ]);

            // تحديث حالة الدفع
            $purchaseInvoice->updatePaymentStatus();

            // جمع معرفات العناصر الحالية للفاتورة
            $existingItemIds = $purchaseInvoice->items->pluck('id')->toArray();
            $updatedItemIds = [];

            // تحديث أو إنشاء عناصر الفاتورة
            foreach ($validated['items'] as $itemData) {
                if (isset($itemData['id']) && in_array($itemData['id'], $existingItemIds)) {
                    // تحديث العنصر الموجود
                    $item = PurchaseInvoiceItem::find($itemData['id']);
                    $item->update([
                        'item_id' => $itemData['item_id'],
                        'purchase_order_item_id' => $itemData['purchase_order_item_id'] ?? null,
                        'quantity' => $itemData['quantity'],
                        'unit_price' => $itemData['unit_price'],
                        'tax_rate' => $itemData['tax_rate'],
                        'tax_amount' => $itemData['tax_amount'],
                        'discount_rate' => $itemData['discount_rate'],
                        'discount_amount' => $itemData['discount_amount'],
                        'total_price' => $itemData['total_price'],
                        'notes' => $itemData['notes'] ?? null,
                    ]);

                    $updatedItemIds[] = $item->id;
                } else {
                    // إنشاء عنصر جديد
                    $item = PurchaseInvoiceItem::create([
                        'purchase_invoice_id' => $purchaseInvoice->id,
                        'item_id' => $itemData['item_id'],
                        'purchase_order_item_id' => $itemData['purchase_order_item_id'] ?? null,
                        'quantity' => $itemData['quantity'],
                        'unit_price' => $itemData['unit_price'],
                        'tax_rate' => $itemData['tax_rate'],
                        'tax_amount' => $itemData['tax_amount'],
                        'discount_rate' => $itemData['discount_rate'],
                        'discount_amount' => $itemData['discount_amount'],
                        'total_price' => $itemData['total_price'],
                        'notes' => $itemData['notes'] ?? null,
                    ]);

                    $updatedItemIds[] = $item->id;
                }
            }

            // حذف العناصر التي لم تعد موجودة في الطلب
            $itemsToDelete = array_diff($existingItemIds, $updatedItemIds);
            if (!empty($itemsToDelete)) {
                PurchaseInvoiceItem::whereIn('id', $itemsToDelete)->delete();
            }

            // إعادة تحميل الفاتورة مع العناصر المحدثة
            $purchaseInvoice->refresh();

            // تحديث المخزون بناءً على التغييرات
            $this->updateInventoryOnEdit($purchaseInvoice, $oldItems);

            // تحديث القيد المحاسبي التلقائي لفاتورة المشتريات
            $journalService = new PurchaseInvoiceJournalService();
            $journalEntry = $journalService->updatePurchaseInvoiceJournalEntry($purchaseInvoice);

            DB::commit();

            $successMessage = 'تم تحديث فاتورة المشتريات وتحديث المخزون بنجاح';
            if ($journalEntry) {
                $successMessage .= ' وتم تحديث القيد المحاسبي التلقائي برقم: ' . $journalEntry->entry_number;
            } else {
                $successMessage .= ' ولكن فشل تحديث القيد المحاسبي التلقائي';
            }

            return redirect()->route('inventory.purchase-invoices.show', $purchaseInvoice)
                ->with('success', $successMessage);
        } catch (\Exception $e) {
            DB::rollBack();

            return redirect()->back()
                ->withInput()
                ->with('error', 'حدث خطأ أثناء تحديث فاتورة المشتريات: ' . $e->getMessage());
        }
    }

    /**
     * حذف فاتورة مشتريات
     */
    public function destroy(PurchaseInvoice $purchaseInvoice)
    {
        $this->authorize('delete_purchase_invoices');

        // لا يمكن حذف الفواتير المدفوعة جزئيًا أو كليًا
        if ($purchaseInvoice->payment_status !== 'unpaid') {
            return redirect()->route('inventory.purchase-invoices.show', $purchaseInvoice)
                ->with('error', 'لا يمكن حذف الفواتير التي تم دفعها جزئيًا أو كليًا');
        }

        DB::beginTransaction();

        try {
            // حذف الملف المرفق إذا وجد
            if ($purchaseInvoice->attachment) {
                Storage::disk('public')->delete($purchaseInvoice->attachment);
            }

            // حذف القيد المحاسبي المرتبط بالفاتورة إذا وجد
            $journalEntry = \App\Models\JournalEntry::where('reference_type', 'App\Models\PurchaseInvoice')
                ->where('reference_id', $purchaseInvoice->id)
                ->first();

            if ($journalEntry) {
                // التحقق من أن القيد غير معتمد
                if ($journalEntry->is_approved) {
                    throw new \Exception('لا يمكن حذف الفاتورة لأن القيد المحاسبي المرتبط بها معتمد');
                }

                // إلغاء ترحيل القيد المحاسبي
                if ($journalEntry->is_posted) {
                    $journalEntry->unpost();
                }

                // حذف بنود القيد المحاسبي
                $journalEntry->items()->delete();

                // حذف القيد المحاسبي
                $journalEntry->delete();
            }

            // حذف الفاتورة (سيتم حذف العناصر تلقائيًا بسبب onDelete('cascade'))
            $purchaseInvoice->delete();

            DB::commit();

            return redirect()->route('inventory.purchase-invoices.index')
                ->with('success', 'تم حذف فاتورة المشتريات بنجاح');
        } catch (\Exception $e) {
            DB::rollBack();

            return redirect()->back()
                ->with('error', 'حدث خطأ أثناء حذف فاتورة المشتريات: ' . $e->getMessage());
        }
    }

    /**
     * إنشاء فاتورة مشتريات من أمر شراء
     */
    public function createFromPurchaseOrder(PurchaseOrder $purchaseOrder)
    {
        $this->authorize('create_purchase_invoices');

        // التحقق من أن أمر الشراء تم استلامه جزئيًا أو كليًا
        if (!in_array($purchaseOrder->status, ['partially_received', 'completed'])) {
            return redirect()->route('inventory.purchase-orders.show', $purchaseOrder)
                ->with('error', 'لا يمكن إنشاء فاتورة مشتريات من أمر شراء لم يتم استلامه');
        }

        $purchaseOrder->load(['supplier', 'warehouse', 'items.item']);

        $suppliers = Supplier::orderBy('name')->get();
        $warehouses = Warehouse::orderBy('name')->get();

        return view('inventory.purchase-invoices.create-from-order', compact('purchaseOrder', 'suppliers', 'warehouses'));
    }

    /**
     * عرض نموذج إضافة دفعة لفاتورة مشتريات
     */
    public function showPaymentForm(PurchaseInvoice $purchaseInvoice)
    {
        $this->authorize('manage_purchase_invoice_payments');

        // لا يمكن إضافة دفعات للفواتير المدفوعة بالكامل
        if ($purchaseInvoice->payment_status === 'paid') {
            return redirect()->route('inventory.purchase-invoices.show', $purchaseInvoice)
                ->with('error', 'لا يمكن إضافة دفعات للفواتير المدفوعة بالكامل');
        }

        $purchaseInvoice->load(['supplier', 'payments']);

        return view('inventory.purchase-invoices.add-payment', compact('purchaseInvoice'));
    }

    /**
     * إضافة دفعة لفاتورة مشتريات
     */
    public function addPayment(Request $request, PurchaseInvoice $purchaseInvoice)
    {
        $this->authorize('manage_purchase_invoice_payments');

        // لا يمكن إضافة دفعات للفواتير المدفوعة بالكامل
        if ($purchaseInvoice->payment_status === 'paid') {
            return redirect()->route('inventory.purchase-invoices.show', $purchaseInvoice)
                ->with('error', 'لا يمكن إضافة دفعات للفواتير المدفوعة بالكامل');
        }

        $validated = $request->validate([
            'payment_date' => 'required|date',
            'amount' => [
                'required',
                'numeric',
                'min:0.01',
                function ($attribute, $value, $fail) use ($purchaseInvoice) {
                    $remainingAmount = $purchaseInvoice->remaining_amount;
                    if ($value > $remainingAmount) {
                        $fail('مبلغ الدفعة يتجاوز المبلغ المتبقي من الفاتورة (' . number_format($remainingAmount, 2) . ')');
                    }
                },
            ],
            'payment_method' => 'required|string',
            'reference_number' => 'nullable|string',
            'notes' => 'nullable|string',
            'attachment' => 'nullable|file|mimes:pdf,jpg,jpeg,png|max:2048',
        ]);

        // معالجة الملف المرفق إذا وجد
        $attachmentPath = null;
        if ($request->hasFile('attachment')) {
            $file = $request->file('attachment');
            $fileName = 'payment_' . time() . '_' . Str::random(10) . '.' . $file->getClientOriginalExtension();
            $attachmentPath = $file->storeAs('purchase_invoice_payments', $fileName, 'public');
        }

        DB::beginTransaction();

        try {
            // إنشاء سجل الدفعة
            $payment = $purchaseInvoice->payments()->create([
                'created_by' => auth()->id(),
                'payment_date' => $validated['payment_date'],
                'amount' => $validated['amount'],
                'payment_method' => $validated['payment_method'],
                'reference_number' => $validated['reference_number'] ?? null,
                'notes' => $validated['notes'] ?? null,
                'attachment' => $attachmentPath,
            ]);

            // تحديث حالة الدفع للفاتورة
            $purchaseInvoice->updatePaymentStatus();

            DB::commit();

            return redirect()->route('inventory.purchase-invoices.show', $purchaseInvoice)
                ->with('success', 'تم إضافة الدفعة بنجاح');
        } catch (\Exception $e) {
            DB::rollBack();

            // حذف الملف المرفق في حالة حدوث خطأ
            if ($attachmentPath) {
                Storage::disk('public')->delete($attachmentPath);
            }

            return redirect()->back()
                ->withInput()
                ->with('error', 'حدث خطأ أثناء إضافة الدفعة: ' . $e->getMessage());
        }
    }
}
